package es.upv.dsic.issi.dplfw.tests;

import static org.junit.Assert.assertTrue;
import static org.junit.jupiter.api.Assertions.assertEquals;

import java.io.File;
import java.io.IOException;
import java.util.UUID;

import org.apache.commons.io.FileUtils;
import org.eclipse.emf.cdo.session.CDOSession;
import org.eclipse.emf.cdo.transaction.CDOTransaction;
import org.eclipse.emf.cdo.util.CommitException;
import org.eclipse.emf.cdo.util.ConcurrentAccessException;
import org.eclipse.emf.cdo.view.CDOQuery;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.Test;

import es.upv.dsic.issi.dplfw.core.util.IIEQueryBuilder;
import es.upv.dsic.issi.dplfw.infoelements.InfoElement;
import es.upv.dsic.issi.dplfw.infoelements.InfoelementsFactory;
import es.upv.dsic.issi.dplfw.infoelements.TextIE;
import es.upv.dsic.issi.dplfw.repomanager.CDOSessionUtil;
import es.upv.dsic.issi.dplfw.repomanager.IRepositoryManager;
import es.upv.dsic.issi.dplfw.repomanager.RepositoryLocation;
import es.upv.dsic.issi.dplfw.repomanager.UnknownRepositoryException;

class DplfwRepositoryTest {

	private static final String REPO_NAME = "REPO_NAME";
	private static final String RESOURCE_NAME = "ROOT";
	private static final String TITLE = "TEST";
	private static final String TEMP_DIR = "temp" + EcoreUtil.generateUUID();

	@AfterAll
	public static void cleanupTemp() {
		FileUtils.deleteQuietly(new File(TEMP_DIR));
	}

	
	@Test
	void testCreateSessionUsingRepoManager() throws UnknownRepositoryException, CommitException, IOException {
		final String uuid = UUID.randomUUID().toString();
		final File file = createRandomTempFile();
		
		IRepositoryManager.INSTANCE.addRepository(new RepositoryLocation(uuid, file.toURI()));

		CDOSession session = null;
		try {
			session = IRepositoryManager.INSTANCE.openSession(uuid);
			assertTrue(!session.isClosed());
		} finally {
			close(session);
		}
		IRepositoryManager.INSTANCE.deleteRepository(uuid);
	}

	@Test
	void testCreateSessionRaw() throws UnknownRepositoryException, CommitException, IOException {
		final File file = createRandomTempFile();
		
		CDOSession session = null;
		try {
			session = CDOSessionUtil.createLocalCDOSession(file, REPO_NAME);
			assertTrue(!session.isClosed());
		} finally {
			close(session);
		}
	}
	
	@Test
	void testAddInfoElement() throws UnknownRepositoryException, CommitException, IOException {
		final File file = createRandomTempFile();
		addInfoElement(file);
	}


	@Test
	void testInfoElementSaved() throws UnknownRepositoryException, CommitException, IOException {
		final File file = createRandomTempFile();
		
		{
			addInfoElement(file);
		}
		
		{
			CDOSession session = null;
			try {
				session = CDOSessionUtil.createLocalCDOSession(file, REPO_NAME);
				CDOTransaction transaction = null;
				try {
					transaction = session.openTransaction();
					Resource resource = transaction.getResource(RESOURCE_NAME);
					assertEquals(resource.getContents().size(), 1);
					assertTrue(resource.getContents().get(0) instanceof InfoElement);
					assertEquals(((InfoElement) resource.getContents().get(0)).getTitle(), TITLE);
				} finally {
					close(transaction);
				}
			} finally {
				close(session);
			}
		}
	}
	
	@Test
	void testOclQuery() throws UnknownRepositoryException, CommitException, IOException {
		final File file = createRandomTempFile();
		
		{
			addInfoElement(file);
		}
		
		{
		CDOSession session = null;
			try {
				session = CDOSessionUtil.createLocalCDOSession(file, REPO_NAME);
				CDOTransaction transaction = null;
				try {
					 transaction = session.openTransaction();
					CDOQuery query = transaction.createQuery("ocl",
							"infoelements::InfoElement.allInstances()->select(ie | ie.title = title)->asOrderedSet()->first()");
					query.setParameter("title", TITLE);
					InfoElement ie = query.getResultValue(InfoElement.class);
					assertEquals(TITLE, ie.getTitle());
				} finally {
					close(transaction);
				}
			} finally {
				close(session);
			}
		}
	}

	@Test
	void testHibernateQuery() throws UnknownRepositoryException, CommitException, IOException {
		final File file = createRandomTempFile();
		
		{
			addInfoElement(file);
		}
		
		{
			CDOSession session = null;
			try {
				session = CDOSessionUtil.createLocalCDOSession(file, REPO_NAME);
				CDOTransaction transaction = null;
				try {
					transaction = session.openTransaction();
					String storeType = transaction.getSession().getRepositoryInfo().getStoreType();
					IIEQueryBuilder ieQueryBuilder = IIEQueryBuilder.FACTORY.createQueryFor(storeType)
							.where()
							.is("title", TITLE);
					CDOQuery query = transaction.createQuery(ieQueryBuilder.language(), ieQueryBuilder.build());
					InfoElement ie = query.getResultValue(InfoElement.class);
					assertEquals(TITLE, ie.getTitle());
				} finally {
					close(transaction);
				}
			} finally {
				close(session);
			}
		}
	}

	@Test
	void testHibernateQueryBuilder() throws UnknownRepositoryException, CommitException, IOException {
		final File file = createRandomTempFile();
		
		{
			addInfoElement(file);
		}
		
		{
			CDOSession session = null;
			try {
				session = CDOSessionUtil.createLocalCDOSession(file, REPO_NAME);
				CDOTransaction transaction = null;
				try {
					transaction = session.openTransaction();
					CDOQuery query = transaction.createQuery("hql", "from InfoElement ie where ie.title like :title");
					query.setParameter("title", TITLE);
					InfoElement ie = query.getResultValue(InfoElement.class);
					assertEquals(TITLE, ie.getTitle());
				} finally {
					close(transaction);
				}
			} finally {
				close(session);
			}
		}
	}

	private static File createRandomTempFile() {
		return new File(TEMP_DIR + "/cdo_repo" + EcoreUtil.generateUUID());
	}

	private static void addInfoElement(final File file) throws ConcurrentAccessException, CommitException {
		CDOSession session = null;
		try {
			session = CDOSessionUtil.createLocalCDOSession(file, REPO_NAME);
			CDOTransaction transaction = null;
			try {
				transaction = session.openTransaction();
				TextIE textIE = InfoelementsFactory.eINSTANCE.createTextIE();
				textIE.createUUID();
				textIE.setTitle(TITLE);
				transaction.getOrCreateResource(RESOURCE_NAME).getContents().add(textIE);
				transaction.commit();
			} finally {
				close(transaction);
			}
		} finally {
			close(session);
		}
	}
	
	private static void close(CDOSession session) {
		if (session != null) {
			session.close();
		}
	}

	private static void close(CDOTransaction transaction) {
		if (transaction != null) {
			transaction.close();
		}
	}
}
