package es.upv.dsic.issi.dplfw.repomanager;

import java.io.File;
import java.net.InetAddress;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.eclipse.core.runtime.preferences.IEclipsePreferences;
import org.eclipse.core.runtime.preferences.IEclipsePreferences.IPreferenceChangeListener;
import org.eclipse.core.runtime.preferences.IEclipsePreferences.PreferenceChangeEvent;
import org.eclipse.emf.cdo.session.CDOSession;
import org.eclipse.net4j.connector.ConnectorException;
import org.eclipse.net4j.util.lifecycle.ILifecycle;
import org.eclipse.net4j.util.lifecycle.LifecycleEventAdapter;
import org.osgi.service.prefs.BackingStoreException;
import org.osgi.service.prefs.Preferences;


class RepositoryManager implements IPreferenceChangeListener, IRepositoryManager {

	/**
	 * The internal repositories {@link Map}
	 * It is created by demand in the getRepositories() method.
	 * <code>repositories</code> should be a synchronized {@link Map}
	 */
	private Map<String, RepositoryLocation> repositories;

	/**
	 * List of active sessions
	 */
	private List<CDOSession> activeSessions =
			Collections.synchronizedList(new ArrayList<CDOSession>());

	/**
	 * Set of active sessions
	 */
	private Set<RepositoryRegistryListener> repositoryListeners = 
			Collections.synchronizedSet(new HashSet<RepositoryRegistryListener>());

	/**
	 * Package-only constructor, to be used only to initialize the shared instance in {@link IRepositoryManager} 
	 */
	RepositoryManager() {
		repositories = Collections.synchronizedMap(new LinkedHashMap<String, RepositoryLocation>());
		if (RepositoryManagerPlugin.getDefault() != null) {
			// This code will only run when running inside Eclipse
			IEclipsePreferences prefs = (IEclipsePreferences) RepositoryManagerPlugin.getDefault().getPreferences();
			prefs.addPreferenceChangeListener(this);
			try {
				for(String key : prefs.keys()) {
					String location = prefs.get(key, null);
					repositories.put(key, new RepositoryLocation(key, location));
				}
			} catch (BackingStoreException | URISyntaxException e) {
				throw new RuntimeException(e);
			}
		}
	}
	
	/* (non-Javadoc)
	 * @see es.upv.dsic.issi.dplfw.repomanager.IRepositoryManager#addRepository(es.upv.dsic.issi.dplfw.repomanager.RepositoryLocation)
	 */
	@Override
	public void addRepository(RepositoryLocation repository) {
		repositories.put(repository.getUuid(), repository);
		// If running as an Eclipse plugin, make the change in the repositories persistentvie Preferences
		if (RepositoryManagerPlugin.getDefault() != null) {
			Preferences prefs = RepositoryManagerPlugin.getDefault().getPreferences();
			try {
				prefs.put(repository.getUuid(), repository.toStringURI());
				prefs.flush();
			} catch (BackingStoreException e) {
				e.printStackTrace();
			}
		}
	}

	/* (non-Javadoc)
	 * @see es.upv.dsic.issi.dplfw.repomanager.IRepositoryManager#deleteRepository(java.lang.String)
	 */
	@Override
	public void deleteRepository(String uuid) {
		repositories.remove(uuid);
		// If running as an Eclipse plugin, make the change in the repositories persistent via Preferences
		if (RepositoryManagerPlugin.getDefault() != null) {
			Preferences prefs = RepositoryManagerPlugin.getDefault().getPreferences();
			try {
				prefs.remove(uuid);
				prefs.flush();
			} catch (BackingStoreException e) {
				e.printStackTrace();
			}
		}
	}

	
	/* (non-Javadoc)
	 * @see es.upv.dsic.issi.dplfw.repomanager.IRepositoryManager#isKnownRepository(java.lang.String)
	 */
	@Override
	public boolean isKnownRepository(String uuid) {
		return getRepositoriesMap().get(uuid) != null;
	}

	
	/* (non-Javadoc)
	 * @see es.upv.dsic.issi.dplfw.repomanager.IRepositoryManager#getRepository(java.lang.String)
	 */
	@Override
	public RepositoryLocation getRepository(String uuid) throws UnknownRepositoryException  {
		RepositoryLocation location = null;

		location = getRepositoriesMap().get(uuid);

		if (location != null)
			return location;
		else
			throw new UnknownRepositoryException(uuid);
	}

	
	/* (non-Javadoc)
	 * @see es.upv.dsic.issi.dplfw.repomanager.IRepositoryManager#getRepositoryUUID(java.lang.String)
	 */
	@Override
	public String getRepositoryUUID(String locationURI) throws URISyntaxException, ConnectorException, UnknownHostException  {
		// Check if repository is already registered
		for (RepositoryLocation repositoryLocation : getRepositoriesMap().values()) {
			if (repositoryLocation.toStringURI().equals(locationURI)) {
				return repositoryLocation.getUuid();
			}
		}
		// Repository is not registered
		// Connect to it and get the UUID
		URI uri = new URI(locationURI);
		
		// Check host (May throw UnknowHostException)
		InetAddress.getByName(uri.getHost());

		// Connection data
		String protocol = uri.getScheme();
		String host = uri.getHost();
		int port = uri.getPort();
		String repositoryName = !StringUtils.equals(protocol, "file") && uri.getPath().startsWith("/") ? uri.getPath().substring(1) : uri.getPath();
		String username = "";
		String password = "";
		
		if (StringUtils.isNotBlank(uri.getUserInfo())) { // Login needed
			String userinfo[] = uri.getUserInfo().split(":");
			username = userinfo.length > 0 ? userinfo[0] : "";
			password = userinfo.length > 1 ? userinfo[1] : "";
		}

		// Open session
		CDOSession session = openSessionInternal(protocol, host, port, repositoryName, username, password);
		
	    // Open session and getting UUID
		String uuid = session.getRepositoryInfo().getUUID();
		session.close();
		return uuid;
	}

	
	/* (non-Javadoc)
	 * @see es.upv.dsic.issi.dplfw.repomanager.IRepositoryManager#getRepositories()
	 */
	@Override
	public Collection<RepositoryLocation> getRepositories()  {
		return Collections.unmodifiableCollection(getRepositoriesMap().values());
	}

	/* (non-Javadoc)
	 * @see es.upv.dsic.issi.dplfw.repomanager.IRepositoryManager#getRepositoriesMap()
	 */
	@Override
	public Map<String, RepositoryLocation> getRepositoriesMap() {
		return Collections.unmodifiableMap(repositories);
	}

	/* (non-Javadoc)
	 * @see es.upv.dsic.issi.dplfw.repomanager.IRepositoryManager#openSession(java.lang.String)
	 */
	@Override
	public CDOSession openSession(String uuid) throws UnknownRepositoryException {
		RepositoryLocation location = getRepository(uuid);
		return openSessionInternal(location.getProtocol(), location.getHost(), location.getPort(),
				location.getRepositoryName(), location.getUsername(),
				location.getPassword());
	}

	/**
	 * Opens a new session to the <code>repository</code> located at the given
	 * <code>host</code>:<code>port</code> using the provided login credentials
	 * 
	 * @param host
	 *            the repository hostname
	 * @param host 
	 * @param port
	 *            the listening port
	 * @param repositoryName
	 *            the repository name
	 * @param username
	 *            the username
	 * @param password
	 *            the password
	 * @return the new {@link CDOSession}
	 */

	protected CDOSession openSessionInternal(
			String protocol, String host, int port, String repositoryName, String username, String password) {
		
		final CDOSession session =
				StringUtils.equals("tcp", protocol) ?
						CDOSessionUtil.createRemoteCDOSession(host, port, repositoryName, username, password) :
						StringUtils.equals("file", protocol) ? 
								CDOSessionUtil.createLocalCDOSession(new File(repositoryName), new File(repositoryName).getName()) : null;						
		
		if (!session.isClosed()) {
			activeSessions.add(session);
		}
		
		session.addListener(new LifecycleEventAdapter() {
			@Override
			protected void onDeactivated(ILifecycle lifecycle) {
				activeSessions.remove(session);
			}
		});
		
		return session;
	}

	
	/**
	 * Updates the repositories {@link Map} when the state of the plugin's
	 * {@link Preferences} changes.
	 */
	@Override
	public void preferenceChange(PreferenceChangeEvent event) {

		RepositoryLocation location = null;
		
		if (event.getNewValue() == null) { // Repository has been deleted
			location = repositories.get(event.getKey());
			repositories.remove(event.getKey());
		} else { // Repository has been added or modified
			try {
				location = new RepositoryLocation(event.getKey(), (String) event.getNewValue());
				repositories.put(event.getKey(), location);
			} catch (URISyntaxException e) {
				e.printStackTrace();
			}
		}
		
		// Notify repository listeners
		synchronized (repositoryListeners) {
			for (Iterator<RepositoryRegistryListener> it = repositoryListeners.iterator(); it.hasNext();) {
				RepositoryRegistryListener listener = (RepositoryRegistryListener) it.next();
				
				RepositoryRegistryEvent.Type type = null;
				
				if (event.getNewValue() == null) 
					type = RepositoryRegistryEvent.Type.DELETION;
				else if (event.getOldValue() == null)
					type = RepositoryRegistryEvent.Type.ADDITION;
				else if (event.getOldValue() != event.getNewValue())
					type = RepositoryRegistryEvent.Type.MODIFICATION;
				
				listener.repositoryRegistryChange(new RepositoryRegistryEvent(location, type));
			}
		}
	}

	
	
	/* (non-Javadoc)
	 * @see es.upv.dsic.issi.dplfw.repomanager.IRepositoryManager#getActiveSessions()
	 */
	@Override
	public List<CDOSession> getActiveSessions() {
		return Collections.unmodifiableList(activeSessions);
	}

	/* (non-Javadoc)
	 * @see es.upv.dsic.issi.dplfw.repomanager.IRepositoryManager#addRepositoryRegistryListener(es.upv.dsic.issi.dplfw.repomanager.RepositoryRegistryListener listener)
	 */
	@Override
	public void addRepositoryRegistryListener(RepositoryRegistryListener listener) {
		repositoryListeners.add(listener);
	}

	/* (non-Javadoc)
	 * @see es.upv.dsic.issi.dplfw.repomanager.IRepositoryManager#removeRepositoryRegistryListener(es.upv.dsic.issi.dplfw.repomanager.RepositoryRegistryListener  listener)
	 */
	@Override
	public void removeRepositoryRegistryListener(RepositoryRegistryListener listener) {
		repositoryListeners.remove(listener);
	}
	
	@Override
	protected void finalize() throws Throwable {
		repositoryListeners = null;
		
		synchronized (activeSessions) { // Required synchronized access when iterating
			for (Iterator<CDOSession> it = activeSessions.iterator(); it.hasNext();) {
				CDOSession session = (CDOSession) it.next();
				session.close();
			}
		}
		
		activeSessions = null;
		
		super.finalize();
	}
	
}
