package es.upv.dsic.issi.dplfw.repomanager;

import java.net.URISyntaxException;
import java.net.UnknownHostException;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.cdo.session.CDOSession;
import org.eclipse.net4j.connector.ConnectorException;


public interface IRepositoryManager {
	
	public final static IRepositoryManager INSTANCE = new RepositoryManager();
	
	public final static String DEFAUL_REPOSITORY_UUID = "default";

	/**
	 * Adds the given repository to the repository registry. If the repository
	 * already exists it is updated.
	 * 
	 * @param repository
	 */
	public void addRepository(RepositoryLocation repository);

	/**
	 * Deletes the repository with the given UUID from the repository registry.
	 * 
	 * @param uuid
	 *            the server's uuid
	 */
	public void deleteRepository(String uuid);

	/**
	 * Checks if the repository with the given UUID is in the repository
	 * registry
	 * 
	 * @param uuid
	 *            the server's uuid
	 * @return <code>true</code> if the repository exists and <code>false</code>
	 *         if not.
	 */
	public boolean isKnownRepository(String uuid);

	/**
	 * Gets the {@link RepositoryLocation} with the given UUID from the
	 * repositories registry
	 * 
	 * @param the
	 *            uuid
	 * @return the {@link RepositoryLocation}
	 * @throws UnknownRepositoryException
	 */
	public RepositoryLocation getRepository(String uuid)
			throws UnknownRepositoryException;

	/**
	 * Gets the UUID of the repository at the given <code>locationURI</code> If
	 * the repository is a know repository, the returned UUID is read from the
	 * repositories registry. If the repository does not exist in the registry
	 * this method tries to connect to the given location, and queries its UUID.
	 * 
	 * @param locationURI
	 * @return the UUID
	 * @throws URISyntaxException
	 *             if the format of locationURI is not valid
	 * @throws UnknownHostException
	 *             if the host of the locationURI does not exists
	 * @throws ConnectorException
	 *             if the location is unknown (it is not registered in the
	 *             registry) and it is impossible to connect to it
	 */
	public String getRepositoryUUID(String locationURI)
			throws URISyntaxException, ConnectorException, UnknownHostException;

	/**
	 * Gets the registered repositories
	 * 
	 * @return the collection of repositories
	 */
	public Collection<RepositoryLocation> getRepositories();

	/**
	 * Gets a {@link Map} of repositories. The keys of the Map are the
	 * <code>UUID</code>s.
	 * 
	 * @return the repositories {@link Map}
	 */
	public Map<String, RepositoryLocation> getRepositoriesMap();

	/**
	 * Opens a new session for the repository with the given UUID. Clients are
	 * responsible of closing the returned {@link CDOSession}.
	 * 
	 * @param uuid
	 *            the repository UUI
	 * @return the new {@link CDOSession}
	 * @throws UnknownRepositoryException
	 */
	public CDOSession openSession(String uuid)
			throws UnknownRepositoryException;

	/**
	 * Returns the list of the active sessions opened by this repository manager.
	 * 
	 * @return the list of active {@link CDOSession}s
	 */
	public List<CDOSession> getActiveSessions();


	/**
	 * Removes the given listener 
	 * 
	 * @param listener
	 *            the listener
	 */
	public abstract void removeRepositoryRegistryListener(RepositoryRegistryListener listener);

	/**
	 * Adds the given listener which will be notified when repositories in the
	 * registry change. If the listener already exists it won't be added again.
	 * 
	 * @param listener
	 *            the listener
	 */
	public abstract void addRepositoryRegistryListener(RepositoryRegistryListener listener);

}