package es.upv.dsic.issi.dplfw.repomanager;

import java.io.File;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.eclipse.emf.cdo.net4j.CDONet4jSessionConfiguration;
import org.eclipse.emf.cdo.net4j.CDONet4jUtil;
import org.eclipse.emf.cdo.server.CDOServerUtil;
import org.eclipse.emf.cdo.server.IRepository;
import org.eclipse.emf.cdo.server.IStore;
import org.eclipse.emf.cdo.server.hibernate.CDOHibernateUtil;
import org.eclipse.emf.cdo.server.hibernate.teneo.TeneoUtil;
import org.eclipse.emf.cdo.server.net4j.CDONet4jServerUtil;
import org.eclipse.emf.cdo.session.CDOSession;
import org.eclipse.net4j.Net4jUtil;
import org.eclipse.net4j.acceptor.IAcceptor;
import org.eclipse.net4j.connector.IConnector;
import org.eclipse.net4j.jvm.JVMUtil;
import org.eclipse.net4j.tcp.TCPUtil;
import org.eclipse.net4j.util.container.ContainerUtil;
import org.eclipse.net4j.util.container.IManagedContainer;
import org.eclipse.net4j.util.lifecycle.ILifecycle;
import org.eclipse.net4j.util.lifecycle.LifecycleEventAdapter;
import org.eclipse.net4j.util.security.PasswordCredentialsProvider;

public class CDOSessionUtil {

	
	/**
	 * Opens a new session to the <code>repository</code> located at the given
	 * <code>host</code>:<code>port</code> using the provided login credentials
	 * 
	 * @param host
	 *            the repository hostname
	 * @param port
	 *            the listening port
	 * @param repositoryName
	 *            the repository name
	 * @param username
	 *            the username
	 * @param password
	 *            the password
	 * @return the new {@link CDOSession}
	 */

	public static CDOSession createRemoteCDOSession(
			String host, int port, String repositoryName, String username, String password) {
		
		// Container needed to avoid closing active 
		// connections when a new connection fails
	    final IManagedContainer container = ContainerUtil.createContainer();
	    Net4jUtil.prepareContainer(container); // Register Net4j factories
	    TCPUtil.prepareContainer(container); // Register TCP factories
	    CDONet4jUtil.prepareContainer(container); // Register CDO factories
	    container.activate();

	    
	    CDONet4jSessionConfiguration configuration = CDONet4jUtil.createNet4jSessionConfiguration();
	    
   	    // Configure connection if login is needed 
	    if (StringUtils.isNotBlank(username)) {
		    PasswordCredentialsProvider credentialsProvider = new PasswordCredentialsProvider(username, password);
		    configuration.setCredentialsProvider(credentialsProvider);
			
		    /*
		     * This authentication method is deprecated
		     * 
		    ConnectorCredentialsInjector newConnector = new ConnectorCredentialsInjector(host + ":" + port, credentialsProvider);
			container.addPostProcessor(newConnector);
			*/
	    }

		// Trying to connect
		final IConnector connector = TCPUtil.getConnector(container, host + ":" + port);
	    configuration.setConnector(connector);
	    configuration.setRepositoryName(repositoryName);
	    
	    // Open session 
		CDOSession session = null;
		try {
			session = configuration.openNet4jSession();
			session.addListener(new LifecycleEventAdapter() {
				@Override
				protected void onDeactivated(ILifecycle lifecycle) {
					// Free resources when session is closed
					connector.close();
					container.deactivate();
				}
			});
		} catch (Exception e){
			// Free resources if openSession fails
			e.printStackTrace();
			connector.close();
			container.deactivate();
		}
		return session;
	}
	
	public static CDOSession createLocalCDOSession(File file, String repositoryName) {
		
		IStore store = CDOHibernateUtil.createStore(TeneoUtil.createMappingProvider());
		Map<String, String> props = new HashMap<String, String>();
		props.put(IRepository.Props.OVERRIDE_UUID, "");
		props.put(IRepository.Props.SUPPORTING_AUDITS, "false");
		props.put("hibernate.hbm2ddl.auto", "update");
		props.put("hibernate.dialect", "org.hibernate.dialect.H2Dialect");
		props.put("hibernate.connection.driver_class", "org.h2.Driver");
		props.put("hibernate.connection.url", "jdbc:h2:file:" + file.getAbsolutePath());
		 
		IManagedContainer container = ContainerUtil.createContainer();
		Net4jUtil.prepareContainer(container);
		JVMUtil.prepareContainer(container);
		CDONet4jUtil.prepareContainer(container);
		CDONet4jServerUtil.prepareContainer(container);
		container.activate();
		IRepository repository = CDOServerUtil.createRepository(repositoryName, store, props);
		CDOServerUtil.addRepository(container, repository);

		@SuppressWarnings("unused")
		// It is necessary to get the Acceptor in order to initialize it 
		IAcceptor acceptor = JVMUtil.getAcceptor(container, "default"); //$NON-NLS-1$
		IConnector connector = JVMUtil.getConnector(container, "default"); //$NON-NLS-1$

		CDONet4jSessionConfiguration configuration = CDONet4jUtil.createNet4jSessionConfiguration();
		configuration.setConnector(connector);
		configuration.setRepositoryName(repositoryName);

	    // Open session 
		CDOSession session = null;
		try {
			session = configuration.openNet4jSession();
			session.addListener(new LifecycleEventAdapter() {
				@Override
				protected void onDeactivated(ILifecycle lifecycle) {
					// Free resources when session is closed
					connector.close();
					container.deactivate();
				}
			});
		} catch (Exception e){
			// Free resources if openSession fails
			e.printStackTrace();
			connector.close();
			container.deactivate();
		}
		return session;
	}
}
