package es.upv.dsic.issi.dplfw.repomanager.ui.views;

import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.emf.cdo.CDOAdapter;
import org.eclipse.emf.cdo.CDOObject;
import org.eclipse.emf.cdo.eresource.CDOResource;
import org.eclipse.emf.cdo.session.CDOSession;
import org.eclipse.emf.cdo.transaction.CDOTransaction;
import org.eclipse.emf.cdo.util.CDOUtil;
import org.eclipse.emf.cdo.util.CommitException;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IMenuCreator;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.preference.PreferenceDialog;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.util.LocalSelectionTransfer;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DragSourceAdapter;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.dialogs.PreferencesUtil;
import org.eclipse.ui.part.DrillDownAdapter;
import org.eclipse.ui.part.ViewPart;
import org.eclipse.ui.statushandlers.StatusManager;

import es.upv.dsic.issi.dplfw.core.ui.DplfwUiPlugin;
import es.upv.dsic.issi.dplfw.infoelement.singleeditor.IESingleEditorPlugin;
import es.upv.dsic.issi.dplfw.infoelement.singleeditor.editor.CDOObjectEditorInput;
import es.upv.dsic.issi.dplfw.infoelements.InfoElement;
import es.upv.dsic.issi.dplfw.repomanager.IRepositoryManager;
import es.upv.dsic.issi.dplfw.repomanager.RepositoryLocation;
import es.upv.dsic.issi.dplfw.repomanager.RepositoryManagerPlugin;
import es.upv.dsic.issi.dplfw.repomanager.RepositoryRegistryEvent;
import es.upv.dsic.issi.dplfw.repomanager.RepositoryRegistryListener;
import es.upv.dsic.issi.dplfw.repomanager.UnknownRepositoryException;
import es.upv.dsic.issi.dplfw.repomanager.ui.RepositoryManagerUIPlugin;
import es.upv.dsic.issi.dplfw.repomanager.ui.actions.EditIEViewerAction;
import es.upv.dsic.issi.dplfw.repomanager.ui.model.BusyRepositoryException;
import es.upv.dsic.issi.dplfw.repomanager.ui.model.RepositoryNode;
import es.upv.dsic.issi.dplfw.repomanager.ui.views.providers.RepositoriesLabelProvider;
import es.upv.dsic.issi.dplfw.repomanager.ui.views.providers.ViewContentProvider;
import es.upv.dsic.issi.dplfw.repomanager.ui.wizards.SelectInfoElementWizard;


public class RepositoriesView extends ViewPart {

	private final class RepositoriesViewerDoubleClickListener implements IDoubleClickListener {
		@Override
		public void doubleClick(DoubleClickEvent event) {
			ISelection selection = event.getSelection();
			if (selection instanceof IStructuredSelection) {
				Object element = ((IStructuredSelection) selection).getFirstElement();
				if (element instanceof InfoElement) {
					editIEAction.run();
				} else {
					if (element instanceof RepositoryNode) {
						RepositoryNode node = (RepositoryNode) element;
						if (!node.isConnected())
							connectAction.run();
						// else
						//	disconnectAction.run();
					}
					if (!viewer.getExpandedState(element)) {
						viewer.expandToLevel(element, 1);
					} else {
						viewer.collapseToLevel(element, 1);
					}
				}
			}
		}
	}

	private final class ViewerRepositoryRegistryListener implements RepositoryRegistryListener {
		@Override
		public void repositoryRegistryChange(RepositoryRegistryEvent event) {
			RepositoryLocation location = event.getRepositoryLocation();
			if (event.getType() == RepositoryRegistryEvent.Type.ADDITION) {
				RepositoryNode node = new RepositoryNode(location);
				repositoryNodes.put(location.getUuid(), node);
				viewer.refresh(node);
			} else if (event.getType() == RepositoryRegistryEvent.Type.DELETION) {
				RepositoryNode node = repositoryNodes.get(location.getUuid());
				repositoryNodes.remove(location.getUuid());
				viewer.refresh(node);
			} else if (event.getType() == RepositoryRegistryEvent.Type.MODIFICATION) {
				RepositoryNode node = repositoryNodes.get(location.getUuid());
				viewer.update(node, null);
			} 
			viewer.refresh();
		}
	}

	private final class ViewerRefreshAdapter extends AdapterImpl implements CDOAdapter {
		@Override
		public void notifyChanged(final Notification msg) {
			Display.getDefault().asyncExec(new Runnable() {
				@Override
				public void run() {
					if (!viewer.getTree().isDisposed()) {
						if (msg.getNotifier() instanceof InfoElement && 
								msg.getFeature() != null) {
							viewer.update(msg.getNotifier(), new String[] { ((EStructuralFeature)msg.getFeature()).getName() });
						} else {
							viewer.refresh();
						}
					}
				}
			});
		}
	}

	/**
	 * The ID of the view as specified by the extension.
	 */
	public static final String ID = "es.upv.dsic.issi.dplfw.repomanager.ui.views.RepositoriesView"; //$NON-NLS-1$
	
	
	private ViewerRefreshAdapter adapter = new ViewerRefreshAdapter();
	
	private Action connectAction;
	
	private Action disconnectAction;
	
	private Action manageAction;

	private Action editIEAction;

	private Action createIEAction;

	private Action refreshAction;

	private Action searchAction;
	
	private DrillDownAdapter drillDownAdapter;
	
	private RepositoryRegistryListener listener = new ViewerRepositoryRegistryListener(); 

	private IRepositoryManager repositoryManager;
	
	/**
	 * Map of opened sessions (key = Repository UUID)
	 */
	private Map<String, RepositoryNode> repositoryNodes = new LinkedHashMap<String, RepositoryNode>();
	
	private TreeViewer viewer;

	/**
	 * The constructor.
	 */
	public RepositoriesView() {
		repositoryManager = IRepositoryManager.INSTANCE;
		for (RepositoryLocation location : repositoryManager.getRepositories()) {
			repositoryNodes.put(location.getUuid(), new RepositoryNode(location));
		}
		repositoryManager.addRepositoryRegistryListener(listener);

	}

	private void contributeToActionBars() {
		IActionBars bars = getViewSite().getActionBars();
		fillLocalPullDown(bars.getMenuManager());
		fillLocalToolBar(bars.getToolBarManager());
	}

	/**
	 * This is a callback that will allow us
	 * to create the viewer and initialize it.
	 */
	public void createPartControl(Composite parent) {
		
		viewer = new TreeViewer(parent, SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL | SWT.VIRTUAL);
		
		viewer.setContentProvider(new ViewContentProvider());
		viewer.setLabelProvider(new RepositoriesLabelProvider());
		viewer.setSorter(new ViewerSorter());
		viewer.setInput(repositoryNodes.values());
		
		viewer.setSorter(new RepositoriesViewerSorter());
		
		viewer.addDragSupport(DND.DROP_COPY | DND.DROP_MOVE, 
				new Transfer[]{LocalSelectionTransfer.getTransfer()}, 
				new DragSourceAdapter());
		
		viewer.addDropSupport(DND.DROP_COPY | DND.DROP_MOVE, 
				new Transfer[]{LocalSelectionTransfer.getTransfer()},
				new RepositoriesViewerDropAdapter(viewer)
		);
		
		addDoubleClickListener();
		
		drillDownAdapter = new DrillDownAdapter(viewer);
		
		// Create the help context id for the viewer's control
		PlatformUI.getWorkbench().getHelpSystem().setHelp(viewer.getControl(), "es.upv.dsic.issi.dplfw.repomanager.ui.viewer"); //$NON-NLS-1$
		
		makeActions();
		hookContextMenu();
		contributeToActionBars();
	}

	private void addDoubleClickListener() {
		viewer.addDoubleClickListener(new RepositoriesViewerDoubleClickListener());
	}

	private void fillContextMenu(IMenuManager manager) {

		IStructuredSelection selection = (IStructuredSelection) viewer.getSelection();
		
		// Check if elements are all of the same type
		{
			boolean allSametype = true;
			Class<?> c = selection.getFirstElement().getClass();
			for (Object element : selection.toList()) {
				if (!c.isInstance(element)) {
					allSametype = false;
					break;
				}
			}
			
			// This view's actions
			if (allSametype) {
				if (selection.getFirstElement() instanceof RepositoryNode) {
					int connected = 0, disconnected = 0;;
					for (Object element : selection.toList()) {
						if (((RepositoryNode) element).isConnected()) {
							connected++;
						} else {
							disconnected++;
						}
					}
					
					if (selection.size() == connected) {
						manager.add(disconnectAction);
					} else if (selection.size() == disconnected) { 
						manager.add(connectAction);
					}

					if (selection.size() == 1 && ((RepositoryNode) selection.getFirstElement()).isConnected()) {
						manager.add(searchAction);
					}
					
					
				} else if (selection.getFirstElement() instanceof InfoElement) {
					manager.add(editIEAction);
				}
			}
		}

		// Create IE actions
		if (selection.size() == 1 && 
				Platform.getAdapterManager().getAdapter(selection.getFirstElement(), CDOResource.class) != null) {
			manager.add(new Separator("es.upv.dsic.issi.dplfw.repomanager.ui.popup.new"));
			manager.add(createIEAction);
		}
		
		// Navigation actions 
		manager.add(new Separator());
		drillDownAdapter.addNavigationActions(manager);
		
		// Other plug-ins can contribute there actions here
		manager.add(new Separator(IWorkbenchActionConstants.MB_ADDITIONS));
	}

	private void fillLocalPullDown(IMenuManager manager) {
		manager.add(searchAction);
		manager.add(new Separator());
		manager.add(manageAction);
		manager.add(new Separator());
	}
	
	private void fillLocalToolBar(IToolBarManager manager) {
		manager.add(searchAction);
		manager.add(new Separator());
		manager.add(manageAction);
		manager.add(new Separator());
		drillDownAdapter.addNavigationActions(manager);
		manager.add(refreshAction);
	}

	private void hookContextMenu() {
		MenuManager menuMgr = new MenuManager("#PopupMenu"); //$NON-NLS-1$
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(new IMenuListener() {
			public void menuAboutToShow(IMenuManager manager) {
				RepositoriesView.this.fillContextMenu(manager);
			}
		});
		Menu menu = menuMgr.createContextMenu(viewer.getControl());
		viewer.getControl().setMenu(menu);
		getSite().registerContextMenu(menuMgr, viewer);
	}

	private void makeActions() {
		makeConnectAction();		
		makeDisconnectAction();
		makeManageAction();
		makeEditIEAction();
		makeCreateIEAction();
		makeRefreshAction();
		makeSearchAction();
	}

	private void makeConnectAction() {
		connectAction = new Action() {
			
			public void run() {
				final IStructuredSelection selection = (IStructuredSelection) viewer.getSelection();

				Job connectJob = new Job("Connecting to repositories") {
					@Override
					protected IStatus run(IProgressMonitor monitor) {
						List<?> elements = ((IStructuredSelection) selection).toList();
						monitor.beginTask("Connecting to repositories...", elements.size());
						for (Object element : elements) {
							final RepositoryNode node = (RepositoryNode) element;
							monitor.subTask(String.format("Connecting to %s", node.getLocation().getUuid()));
							try {
								internalConnectRepository(node);
								getSite().getShell().getDisplay().syncExec(new Runnable() {
									@Override
									public void run() {
										viewer.refresh(node);
									}
								});
							} catch (UnknownRepositoryException e) {
								logException(e);
							}
							monitor.worked(1);
						}
						monitor.done();
						return Status.OK_STATUS;
					}
				};
				connectJob.setUser(true);
				connectJob.schedule();
			}
		};
		connectAction.setText("&Connect"); 
		connectAction.setToolTipText("Connect to Repository"); 
		connectAction.setImageDescriptor(ImageDescriptor
				.createFromImage(RepositoryManagerUIPlugin.getDefault()
						.getImageRegistry()
						.get(RepositoryManagerUIPlugin.IMG_ELCL16_CONNECT)));
	}

	private void makeDisconnectAction() {
		disconnectAction = new Action() {
			
			
			public void run() {
				final IStructuredSelection selection = (IStructuredSelection) viewer.getSelection();

				Job connectJob = new Job("Disconnecting repositories") {
					@Override
					protected IStatus run(IProgressMonitor monitor) {
						List<?> elements = ((IStructuredSelection) selection).toList();
						monitor.beginTask("Disconnecting repositories...", elements.size());
						for (Object element : elements) {
							final RepositoryNode node = (RepositoryNode) element;
							monitor.subTask(String.format("Disconnecting from %s", node.getLocation().getUuid()));
							try {
								node.disconnect();
								getSite().getShell().getDisplay().syncExec(new Runnable() {
									@Override
									public void run() {
										viewer.refresh(node);
									}
								});
							} catch (BusyRepositoryException e) {
								getSite().getShell().getDisplay().syncExec(new Runnable() {
									@Override
									public void run() {
										boolean forceDisconnect = MessageDialog.openQuestion(getSite().getShell(), 
												"Unable to close repository",
												String.format("The repository \"%s\" has open connections.\nClose them anyway?", node.getName()));
										if (forceDisconnect) {
											node.forceDisconnect();
											viewer.refresh(node);
										}
									}
								});
							}
							monitor.worked(1);
						}
						monitor.done();
						return Status.OK_STATUS;
					}
				};
				connectJob.setUser(true);
				connectJob.schedule();
			}
		};
		
		disconnectAction.setText("&Disconnect"); 
		disconnectAction.setToolTipText("Disconnect from Repository"); 
		disconnectAction.setImageDescriptor(ImageDescriptor
				.createFromImage(RepositoryManagerUIPlugin.getDefault()
						.getImageRegistry()
						.get(RepositoryManagerUIPlugin.IMG_ELCL16_DISCONNECT)));
	}

	private void makeManageAction() {
		manageAction = new Action() {
			public void run() {
				PreferenceDialog dialog = PreferencesUtil.createPreferenceDialogOn(getSite().getShell(),
						RepositoryManagerUIPlugin.PREFERENCES_REPOSITORIES_ID, 
						new String[] { DplfwUiPlugin.PREFERENCES_ROOT_ID, 
										RepositoryManagerUIPlugin.PREFERENCES_REPOSITORIES_ID }, 
						null);
				dialog.open();
			}
		};
		manageAction.setText("&Manage Repositories");
		manageAction.setToolTipText("Add and Remove Repositories"); 
		manageAction.setImageDescriptor(ImageDescriptor
				.createFromImage(RepositoryManagerUIPlugin.getDefault()
						.getImageRegistry()
						.get(RepositoryManagerUIPlugin.IMG_ELCL16_MANAGE)));
	}
	
	private void makeEditIEAction() {
		editIEAction = new EditIEViewerAction(viewer);
		editIEAction.setText("&Edit");
		editIEAction.setToolTipText("Opens the Info Element with the default editor");
		editIEAction.setImageDescriptor(ImageDescriptor
				.createFromImage(RepositoryManagerUIPlugin.getDefault()
						.getImageRegistry()
						.get(RepositoryManagerUIPlugin.IMG_ELCL16_EDIT)));

		
	}
	
	private void makeRefreshAction() {
		refreshAction = new Action() {
			@Override
			public void run() {
				getSite().getShell().getDisplay().asyncExec(new Runnable() {
					@Override
					public void run() {
						viewer.refresh(true);
					}
				});
			}

		};
		refreshAction.setText("&Refresh");
		refreshAction.setToolTipText("Forces the viewer to be refreshed");
		refreshAction.setImageDescriptor(ImageDescriptor
				.createFromImage(RepositoryManagerUIPlugin.getDefault()
						.getImageRegistry()
						.get(RepositoryManagerUIPlugin.IMG_ELCL16_REFRESH)));
	}
	
	private void makeSearchAction() {
		searchAction = new Action() {
			@Override
			public void run() {

				// Get the selected RepositoryNode
				RepositoryLocation location = null;
				StructuredSelection selection = (StructuredSelection) viewer.getSelection();
				
				if (!selection.isEmpty() && selection.getFirstElement() instanceof RepositoryNode) {
					location = ((RepositoryNode) selection.getFirstElement()).getLocation();
				}
				
				// Create the wizard using the location (may be null)
				SelectInfoElementWizard wizard = new SelectInfoElementWizard(location);
				WizardDialog wizardDialog = new WizardDialog(getSite().getShell(), wizard) {
					{
						// Initialization to allow the wizard to be non-modal
						setShellStyle(SWT.SHELL_TRIM);
					}
				};
				
				wizardDialog.create();
				wizardDialog.setBlockOnOpen(true);
				
				// Process the result!
				if (wizardDialog.open() == WizardDialog.OK) {
					URI uri = wizard.getInfoElementUri();

					try {
						// We get a connected node
						RepositoryNode node = repositoryNodes.get(uri.host());
						if (node == null) 
							return;
						if (!node.isConnected()) {
							boolean connect = MessageDialog.openQuestion(getSite().getShell(), 
									"Connect to repository?",
									String.format("Repository '%s' is disconnected. " +
												"Do yo want to connect to it now?",
									node.getLocation().toStringURI()));
								if (connect) {
									internalConnectRepository(node);
								}
								viewer.update(node, null);
						}
						// And the live element
						EObject element = node.getRootResource().getEObject(uri.fragment());
						
						// We have to gather all the parents in order to show them in the viewer
						LinkedList<Object> elements = new LinkedList<Object>();
						EObject parent = element;

						while (parent != null) {
							elements.addFirst(parent);
							if (parent.eContainer() != null) {
								parent = parent.eContainer();
							} else if (parent.eResource() != null && 
									parent.eResource() != parent &&
									parent.eResource() != node.getRootResource() &&
									parent.eResource() instanceof EObject) {
								parent = (EObject) parent.eResource();
							} else {
								parent = null;
							}
						};
						
						elements.addFirst(node);
						// We expand all the hierarchy
						for (int i = 0; i < elements.size(); i++) {
							viewer.expandToLevel(elements.get(i), 1);
						}
						// and select the element
						viewer.setSelection(new StructuredSelection(element));
					} catch (UnknownRepositoryException e) {
						logException(e);
					}
				}
			}
		};
		searchAction.setText("&Search...");
		searchAction.setToolTipText("Searches for a Specific Info Element in a repository");
		searchAction.setImageDescriptor(ImageDescriptor
				.createFromImage(RepositoryManagerUIPlugin.getDefault()
						.getImageRegistry()
						.get(RepositoryManagerUIPlugin.IMG_ELCL16_SEARCH)));
	}
	
	private void makeCreateIEAction() {
		createIEAction = new Action("New InfoElement", Action.AS_DROP_DOWN_MENU) {
			public void run() {
			}
		};
		createIEAction.setText("New InfoElement");
		createIEAction.setToolTipText("Creates a new InfoElement in the selected repository location");
		createIEAction.setImageDescriptor(PlatformUI.getWorkbench().getSharedImages().
			getImageDescriptor(ISharedImages.IMG_OBJ_ADD));
		
		
		createIEAction.setMenuCreator(new IMenuCreator() {
			private Menu menuInControl;
			private Menu menuInMenu;
			
			@Override
			public Menu getMenu(Menu parent) {
				if (menuInMenu == null || menuInMenu.isDisposed()) {
					menuInMenu = new Menu(parent);
					populateMenu(menuInMenu);
				}
				return menuInMenu;
			}
			
			@Override
			public Menu getMenu(Control parent) {
				if (menuInControl == null || menuInControl.isDisposed()) {
					menuInControl = new Menu(parent);
					populateMenu(menuInControl);
				}
				return menuInControl;
			}

			private void populateMenu(Menu menu) {
				IExtensionRegistry registry = Platform.getExtensionRegistry();
				IConfigurationElement[] extensions = registry.getConfigurationElementsFor(
						RepositoryManagerUIPlugin.IECREATION_EXT_POINT_ID);
				
				
				MenuItem menuItem = null;

				for (final IConfigurationElement configElement : extensions) {
					menuItem = new MenuItem(menu, SWT.PUSH);
					menuItem.setText(configElement.getAttribute(RepositoryManagerUIPlugin.IECREATION_EXT_POINT_LABEL));

					try {
						CDOObject dummyElement = (InfoElement) configElement.createExecutableExtension(
								RepositoryManagerUIPlugin.IECREATION_EXT_POINT_IECLASS);

						if (configElement.getAttribute(RepositoryManagerUIPlugin.IECREATION_EXT_POINT_ICON) == null) {
							Image image = ((ILabelProvider)viewer.getLabelProvider()).getImage(dummyElement);
							if (image != null)
								menuItem.setImage(image);
						} else {
							menuItem.setImage(new Image(getSite().getShell().getDisplay(), 
									getClass().getResourceAsStream("/" + 
											configElement.getAttribute(RepositoryManagerUIPlugin.IECREATION_EXT_POINT_ICON))));
						}
						menuItem.addSelectionListener(new SelectionAdapter() {
							@Override
							public void widgetSelected(SelectionEvent e) {
								
								try {
									InfoElement element = (InfoElement) configElement.createExecutableExtension(
											RepositoryManagerUIPlugin.IECREATION_EXT_POINT_IECLASS);
									element.createUUID();
									
									IStructuredSelection selection = (IStructuredSelection) viewer.getSelection();
									CDOResource resource = (CDOResource) Platform.getAdapterManager().getAdapter(selection.getFirstElement(), CDOResource.class);
									if (resource != null) { 
										IWorkbenchPage page = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage();
										CDOSession session = CDOUtil.getSession(resource);
										CDOTransaction transaction = session.openTransaction();
										resource = transaction.getObject(resource);
										try {
											resource.getContents().add(element);
											transaction.commit();
											CDOObjectEditorInput input = new CDOObjectEditorInput(
													transaction,
													resource.getPath(), 
													resource.getURIFragment(element),
													true);
											page.openEditor(input, IESingleEditorPlugin.EDITOR_ID);
										} catch (CommitException ex) {
											transaction.rollback();
											logException(ex);
										} catch (CoreException ex) {
											logException(ex);
										} finally {
											// Transaction can't be closed because it is used in the Editor Input
											// As the editor input owns the transaction, it closes it when disposed					
										}
									}
								} catch (CoreException e1) {
									logException(e1);
								}
							}
						});
					} catch (CoreException e) {
						logException(e);
					}
				}
			}
			
			@Override
			public void dispose() {
				if (menuInControl != null) {
					menuInControl.dispose();
					menuInControl = null;
				}
				if (menuInMenu != null) {
					menuInMenu.dispose();
					menuInMenu = null;
				}
			}
		});
	}

	private void internalConnectRepository(final RepositoryNode node) throws UnknownRepositoryException {
		node.connect();
		node.setEObjectAdapter(adapter);
	}
	
	/**
	 * Passing the focus request to the viewer's control.
	 */
	public void setFocus() {
		viewer.getControl().setFocus();
	}

	void logException(Exception e) {
		StatusManager.getManager().handle(
				new Status(Status.ERROR, RepositoryManagerUIPlugin.PLUGIN_ID, e.getLocalizedMessage(), e),
				StatusManager.BLOCK | StatusManager.LOG);
	}
	
	@Override
	public void dispose() {
		for (RepositoryNode node : repositoryNodes.values()) {
			if (node.isConnected()) {
				node.forceDisconnect();
			}
		}
		repositoryManager.removeRepositoryRegistryListener(listener);
		repositoryNodes.clear();
		super.dispose();
	}

}