package es.upv.dsic.issi.dplfw.repomanager.ui.preferences;

import java.lang.reflect.InvocationTargetException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.IInputValidator;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.preference.PreferencePage;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.net4j.connector.ConnectorException;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPreferencePage;

import es.upv.dsic.issi.dplfw.repomanager.IRepositoryManager;
import es.upv.dsic.issi.dplfw.repomanager.RepositoryLocation;
import es.upv.dsic.issi.dplfw.repomanager.ui.RepositoryManagerUIPlugin;

/**
 * This class represents a preference page that
 * is contributed to the Preferences dialog.
 */

public class RepositoriesPreferencePage
	extends PreferencePage
	implements IWorkbenchPreferencePage {

	private final class RepositoriesMapContentProvider implements
			IStructuredContentProvider {
		@Override
		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		}

		@Override
		public void dispose() {
		}

		@Override
		public Object[] getElements(Object inputElement) {
			return ((Map<?, ?>)inputElement).values().toArray();
		}
	}

	private final class RemoveRepoSelectionAdapter extends SelectionAdapter {
		@Override
		public void widgetSelected(SelectionEvent e) {
			IStructuredSelection selection = (IStructuredSelection) repositoriesViewer.getSelection();
			if (!selection.isEmpty()) {
				RepositoryLocation repositoryLocation = (RepositoryLocation) selection.getFirstElement();
				repositories.remove(repositoryLocation.getUuid());
				repositoriesViewer.remove(repositoryLocation);
				repositoriesViewer.refresh(repositoryLocation);
			}
		}
	}

	private final class NewRepoSelectionListener extends SelectionAdapter {
		
		private final class RetrieveUuidRunnable implements IRunnableWithProgress {
			private String uuid;
			private String newLocationURI;
			private Throwable throwable;
			private boolean success = false;
			
			private RetrieveUuidRunnable(String newLocationURI) {
				this.newLocationURI = newLocationURI;
			}

			@Override
			public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
				monitor.beginTask("Retrieving repository UUID...", IProgressMonitor.UNKNOWN);
				 try {
					 success = false;
					 uuid = IRepositoryManager.INSTANCE.getRepositoryUUID(newLocationURI);
					 success = true;
				} catch (Exception e) {
					this.throwable = e;
				}
			}
			
			public String getUuid() {
				return uuid;
			}
			
			public Throwable getThrowable() {
				return throwable;
			}
			
			public boolean success() {
				return success;
			}
		}

		@Override
		public void widgetSelected(SelectionEvent e) {
			String initialValue = "";
			IStructuredSelection selection = (IStructuredSelection) repositoriesViewer.getSelection();
			if (!selection.isEmpty()) {
				RepositoryLocation repositoryLocation = (RepositoryLocation) selection.getFirstElement();
				initialValue = repositoryLocation.toStringURI();
			}

			InputDialog inputDialog = new InputDialog(getShell(), "Add repository", "Location:", initialValue, new IInputValidator() {
				
				@Override
				public String isValid(String newText) {
					try {
						if (StringUtils.isBlank(newText)) {
							return "Expected syntax: protocol://user:password@hostname:port/repository_name\rExample: tcp://user:user@localhost:12036/repo";
						}
						URI uri = new URI(newText);
						if (uri.getScheme() == null) 
							return "Unknown protocol";
						else if (!StringUtils.equals(uri.getScheme(), "file") && uri.getHost() == null) 
							return "Expected a host location";
						else if (uri.getPath() == null) 
							return "Expected a repository location";
						else if (StringUtils.equals(uri.getScheme(), "file") && !uri.getPath().startsWith("/")) 
							return "Expected an absolute path";
					} catch (URISyntaxException e) {
						return e.getLocalizedMessage();
					}
					return null;
				}
			});
			if (inputDialog.open() == InputDialog.OK) {

				String newLocationURI = inputDialog.getValue();
				
				try {
					String uuid = retrieveRepoUuid(newLocationURI);
					RepositoryLocation existingLocation = repositories.get(uuid);
					
					boolean addRepository = true;

					// Check if repository already exists
					if (existingLocation != null) {
						// Check if repository already exists with diferent URI
						if (!existingLocation.toStringURI().equals(newLocationURI)) {
							// Repository already exists with different info. Overwrite?
							addRepository = MessageDialog.openQuestion(
									getShell(), 
									"Overwrite?",
									String.format("Repository with UUID {%s} already exists with URI \"%s\".\nReplace it with \"%s\"?",
											uuid, existingLocation.toStringURI(), newLocationURI));
						} else {
							// Repository already exists with the same information
							// Update is not needed.
							addRepository = false;
							MessageDialog.openInformation(getShell(), 
									"Repository already exists",
									String.format("Repository with UUID {%s} already exists",uuid));
						}
					}
					if (addRepository) {
						RepositoryLocation repositoryLocation = new RepositoryLocation(uuid, newLocationURI);
						repositories.put(uuid, repositoryLocation);
						repositoriesViewer.refresh();
					}
				} catch (UnknownHostException e1) {
					MessageDialog.openError(getShell(), "Error",
					String.format("Unable to connect to \"%s\".\nUnknown host: %s", newLocationURI, e1.getLocalizedMessage()));
				} catch (ConnectorException e1) {
					 MessageDialog.openError(getShell(), "Error",
					 String.format("Unable to connect to \"%s\".\n%s", newLocationURI, e1.getLocalizedMessage()));
				} catch (Throwable e1) {
					 MessageDialog.openError(getShell(), "Error",
					 String.format("Unable to connect to \"%s\".\n%s", newLocationURI, e1.toString()));
				}
			}
			
		}

		private String retrieveRepoUuid(final String newLocationURI) throws Throwable {
			
			ProgressMonitorDialog dialog = new ProgressMonitorDialog(getShell());
			
			RetrieveUuidRunnable retrieveUuidRunnable = new RetrieveUuidRunnable(newLocationURI);
				dialog.run(true, false, retrieveUuidRunnable);
				if (retrieveUuidRunnable.success()) {
					return retrieveUuidRunnable.getUuid();
				} else {
					throw retrieveUuidRunnable.getThrowable(); 
				}
		}
	}


	private HashMap<String, RepositoryLocation> repositories = new LinkedHashMap<String, RepositoryLocation>();
	private TableViewer repositoriesViewer;
	private Button addRepoButton;
	private Button removeRepoButton;


	public RepositoriesPreferencePage() {
		setPreferenceStore(RepositoryManagerUIPlugin.getDefault().getPreferenceStore());
		setDescription("DPLfw available repositories");
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.IWorkbenchPreferencePage#init(org.eclipse.ui.IWorkbench)
	 */
	public void init(IWorkbench workbench) {
		loadRepositories();
	}

	private void loadRepositories() {
		repositories.clear();
		repositories.putAll(IRepositoryManager.INSTANCE.getRepositoriesMap());
	}


	@Override
	protected Control createContents(Composite parent) {

		Composite topComposite = new Composite(parent, SWT.NONE);
		GridLayout topLayout = new GridLayout(2, false);
		topLayout.marginWidth = 0;
		topLayout.marginHeight = 0;
		
		GridData layoutData = new GridData();
		layoutData.grabExcessHorizontalSpace = true;
		layoutData.grabExcessVerticalSpace = true;
		layoutData.horizontalAlignment = SWT.FILL;
		layoutData.verticalAlignment = SWT.FILL;
		layoutData.minimumHeight = 200;
		layoutData.heightHint = 200;
		layoutData.minimumWidth = 400;
		
		topComposite.setLayout(topLayout);
		topComposite.setLayoutData(layoutData);
		
		repositoriesViewer = new TableViewer(topComposite, SWT.FULL_SELECTION | SWT.BORDER | SWT.SINGLE);
		createColumns(repositoriesViewer);
		repositoriesViewer.getTable().setHeaderVisible(true);
		repositoriesViewer.setContentProvider(new RepositoriesMapContentProvider());
		repositoriesViewer.setInput(repositories);
		repositoriesViewer.addSelectionChangedListener(new ISelectionChangedListener() {
			@Override
			public void selectionChanged(SelectionChangedEvent event) {
				removeRepoButton.setEnabled(
						!event.getSelection().isEmpty() 
						&& !StringUtils.equals(
							IRepositoryManager.DEFAUL_REPOSITORY_UUID, 
							((RepositoryLocation) ((IStructuredSelection) event.getSelection()).getFirstElement()).getUuid())
						);
			}
		});
		repositoriesViewer.getControl().setLayoutData(layoutData);
		
		
		Composite buttonsComposite = new Composite(topComposite, SWT.NONE);
		GridData gridData = new GridData(100, SWT.DEFAULT);
		gridData.grabExcessVerticalSpace = false;
		gridData.verticalAlignment = SWT.BEGINNING;
		buttonsComposite.setLayoutData(gridData);
		FillLayout fillLayout = new FillLayout(SWT.VERTICAL);
		fillLayout.spacing = 5;
		buttonsComposite.setLayout(fillLayout);
		
		addRepoButton = new Button(buttonsComposite, SWT.PUSH);
		addRepoButton.setText("&New...");
		addRepoButton.addSelectionListener(new NewRepoSelectionListener());
		
		removeRepoButton = new Button(buttonsComposite, SWT.PUSH);
		removeRepoButton.setText("&Remove");
		removeRepoButton.addSelectionListener(new RemoveRepoSelectionAdapter());
		removeRepoButton.setEnabled(false);

		return topComposite;
	}
	
	// This will create the columns for the table
	private void createColumns(final TableViewer viewer) {
		String[] titles = { "UUID", "Schema", "User", "Host", "Port", "Repository Path" };
		int[] bounds = { 140, 60, 50, 70, 50, 200 };

		TableViewerColumn col = createTableViewerColumn(viewer, titles[0], bounds[0], 0);
		col.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(Object element) {
				RepositoryLocation location = (RepositoryLocation) element;
				return location.getUuid();
			}
			@Override
			public Image getImage(Object element) {
				return RepositoryManagerUIPlugin.getDefault().getImageRegistry().get(
						RepositoryManagerUIPlugin.IMG_OBJ16_REPOSITORY);
			}
		});

		col = createTableViewerColumn(viewer, titles[1], bounds[1], 1);
		col.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(Object element) {
				RepositoryLocation location = (RepositoryLocation) element;
				return location.getProtocol();

			}
		});

		col = createTableViewerColumn(viewer, titles[2], bounds[2], 2);
		col.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(Object element) {
				RepositoryLocation location = (RepositoryLocation) element;
				return StringUtils.defaultIfEmpty(location.getUsername(), "-");

			}
		});
		
		col = createTableViewerColumn(viewer, titles[3], bounds[3], 3);
		col.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(Object element) {
				RepositoryLocation location = (RepositoryLocation) element;
				return StringUtils.defaultIfEmpty(location.getHost(), "-");

			}
		});

		col = createTableViewerColumn(viewer, titles[4], bounds[4], 4);
		col.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(Object element) {
				RepositoryLocation location = (RepositoryLocation) element;
				return location.getPort() >= 0 ? String.valueOf(location.getPort()) : "-";
			}
		});

		col = createTableViewerColumn(viewer, titles[5], bounds[5], 5);
		col.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(Object element) {
				RepositoryLocation location = (RepositoryLocation) element;
				return String.valueOf(location.getRepositoryName());
			}
		});

	}

	private TableViewerColumn createTableViewerColumn(TableViewer viewer, String title, int bound, final int colNumber) {
		final TableViewerColumn viewerColumn = new TableViewerColumn(viewer, SWT.NONE);
		final TableColumn column = viewerColumn.getColumn();
		column.setText(title);
		column.setWidth(bound);
		column.setResizable(true);
		column.setMoveable(true);
		return viewerColumn;

	}
	
	
	@Override
	public boolean performOk() {
		IRepositoryManager manager = IRepositoryManager.INSTANCE;
		List<String> deletedRepositoriesUuids = new ArrayList<String>();

		// First we collect the repositories that have been removed from the viewer
		for (RepositoryLocation repositoryLocation : manager.getRepositories()) {
			if (!repositories.containsKey(repositoryLocation.getUuid())) {
				deletedRepositoriesUuids.add(repositoryLocation.getUuid());
			}
		}
		// Second, we effectively delete the repositories
		for (String uuid :  deletedRepositoriesUuids) {
			manager.deleteRepository(uuid);
		}
		
		// Third, we add/update the remaining repositories
		for (RepositoryLocation repositoryLocation : repositories.values()) {
			manager.addRepository(repositoryLocation);
		}
		
		return super.performOk();
	}

	@Override
	protected void performDefaults() {
		loadRepositories();
		repositoriesViewer.refresh();
		super.performDefaults();
	}
}