package es.upv.dsic.issi.dplfw.om.credentialsmanager.internal;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.util.Collections;
import java.util.UUID;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;

import es.upv.dsic.issi.dplfw.om.Actor;
import es.upv.dsic.issi.dplfw.om.OmFactory;
import es.upv.dsic.issi.dplfw.om.Organization;
import es.upv.dsic.issi.dplfw.om.Unit;
import es.upv.dsic.issi.dplfw.om.User;
import es.upv.dsic.issi.dplfw.om.credentialsmanager.CredentialsManagerPlugin;
import es.upv.dsic.issi.dplfw.om.credentialsmanager.ICredentialsManager;
import es.upv.dsic.issi.dplfw.om.credentialsmanager.exceptions.InvalidLocationException;
import es.upv.dsic.issi.dplfw.om.credentialsmanager.exceptions.NullLocationException;
import es.upv.dsic.issi.dplfw.om.credentialsmanager.exceptions.UninitializedLocationException;

public class LocalFileCredentialsManager implements ICredentialsManager {

	private URI location;
	
	private Organization organization;
	
	private Resource resource;
	
	@Override
	public void setLocation(URI uri) throws InvalidLocationException {
		if (uri != null) {
			this.location = uri;
			if (!fileExists()) {
				throw new InvalidLocationException(uri);
			}
		} else {
			this.location = CredentialsManagerPlugin.getDefault().
					getStateLocation().append("localcredentials.om").toFile().toURI();
		}
	}

	@Override
	public void loadLocation() throws NullLocationException, UninitializedLocationException, IOException {
		if (location == null) {
			throw new NullLocationException();
		} else {
			ResourceSet resourceSet = new ResourceSetImpl();
			
			if (fileExists()) {
				resource = resourceSet.getResource(
						org.eclipse.emf.common.util.URI.createURI(location.toString()), true);
			}
			
			resource.load(Collections.emptyMap());
			
			if (!resource.getContents().isEmpty() &&
					resource.getContents().get(0) instanceof Organization) {
				organization = (Organization) resource.getContents().get(0);
			} else {
				throw new UninitializedLocationException(location);
			}
		}
	}
	
	@Override
	public void initializeLocation() throws IOException {
		ResourceSet resourceSet = new ResourceSetImpl();
		
		// If the local credentials file does not exist we must create it first
		if (fileExists()) {
			resource = resourceSet.getResource(
					org.eclipse.emf.common.util.URI.createURI(location.toString()), true);
			resource.load(Collections.emptyMap());
		} else {
			resource = resourceSet.createResource(
					org.eclipse.emf.common.util.URI.createURI(location.toString()));
		}
		
		// If the resource is empty we must create an empty organization
		if (!resource.getContents().isEmpty()) {
			resource.getContents().clear();
		}
		
		organization = OmFactory.eINSTANCE.createOrganization();
		resource.getContents().add(organization);
		resource.save(Collections.emptyMap());
	}
	
	
	private boolean fileExists() {
		File file = new File(location);
		return file.exists();
	}
	
	protected User getUser(UUID uuid) {
		return (User) resolveActor(uuid);
	}
	
	@Override
	public boolean isValidCredential(UUID uuid, String password) {
		return isValidCredential(getUser(uuid),password);
	}

	@Override
	public boolean isValidCredential(User user, String password) {
		return user.isValidPassword(password);
	}
	
	@Override
	public String resolveActorName(UUID uuid) {
		if (resolveActor(uuid) == null) {
			return null;
		}
		return resolveActor(uuid).getName();
	}
	
	@Override
	public URI getLocation() {
		return location;
	}

	@Override
	public Organization getOrganization() {
		return organization;
	}

	@Override
	public Actor resolveActor(UUID uuid) {
		if (uuid != null) {
			EObject eObject = resource.getEObject(uuid.toString());
			if (eObject instanceof Actor) { 
				return (Actor) eObject;
			}
		}
		return null;
	}

	@Override
	public User resolveUser(String login) {
		for (User user : getOrganization().getUsers()) {
			if (user.getLogin().equals(login)) {
				return user;
			}
		}
		return null;
	}
	
	@Override
	public boolean isIncluded(Actor actor1, Actor actor2) {
		if (actor1 == null || actor2 == null)
			return false;
		
		if (actor1.getUuid().equals(actor2.getUuid())) {
			return true;
		} else if (actor2 instanceof Unit) {
			Unit unit = (Unit) actor2;
			if (unit.getUsers().contains(actor1)) {
				return true;
			}
		}
		return false;
	}
}
