package es.upv.dsic.issi.dplfw.om.credentialsmanager;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.Platform;
import org.osgi.service.prefs.BackingStoreException;
import org.osgi.service.prefs.Preferences;

import es.upv.dsic.issi.dplfw.om.Actor;
import es.upv.dsic.issi.dplfw.om.OmFactory;
import es.upv.dsic.issi.dplfw.om.Organization;
import es.upv.dsic.issi.dplfw.om.User;
import es.upv.dsic.issi.dplfw.om.credentialsmanager.exceptions.InvalidLocationException;
import es.upv.dsic.issi.dplfw.om.credentialsmanager.exceptions.NullLocationException;
import es.upv.dsic.issi.dplfw.om.credentialsmanager.exceptions.UninitializedLocationException;
import es.upv.dsic.issi.dplfw.om.credentialsmanager.exceptions.UnknownCredentialsManagerType;

public class CredentialsManagersRegistry {

	private static final String PREF_NODE_REGISTERED_MANAGERS = "PREF_NODE_REGISTERED_MANAGERS";
	private static final String PREF_ACTIVE_MANAGER_TYPE = "PREF_ACTIVE_MANAGER_TYPE";
	private static final String PREF_ACTIVE_MANAGER_LOCATION = "PREF_ACTIVE_MANAGER_LOCATION";

	public static CredentialsManagersRegistry eINSTANCE;
	private List<CredentialsManagerType> registeredTypes;
	private ConcurrentHashMap<String, ICredentialsManager> registeredManagers;
	
	private ICredentialsManager activeManager = null;
	
	public CredentialsManagersRegistry() throws CoreException, BackingStoreException,
					InvalidLocationException, URISyntaxException, UnknownCredentialsManagerType,
					NullLocationException, IOException, UninitializedLocationException {
		eINSTANCE = this;
		loadRegisteredTypes();
		loadRegisteredManagers();
		loadActiveManager();
	}
	
    private void loadRegisteredTypes() throws CoreException {
		IConfigurationElement[] elements = Platform.getExtensionRegistry().getConfigurationElementsFor(
				CredentialsManagerPlugin.CREDENTIALS_MANAGER_EXT_ID);
		registeredTypes = new ArrayList<CredentialsManagerType>();
		for (IConfigurationElement element : elements) {
			registeredTypes.add(new CredentialsManagerType(element));
		}
    }

	private void loadRegisteredManagers() throws BackingStoreException, URISyntaxException,
				UnknownCredentialsManagerType, CoreException, NullLocationException, IOException {

		Preferences topNode = getPreferencesNode();
		registeredManagers = new ConcurrentHashMap<String, ICredentialsManager>();
		for (String typeId : topNode.childrenNames()) {
			for (String location : topNode.node(typeId).keys()) {
				try {
					registeredManagers.put(topNode.node(typeId) + location, createManager(typeId, new URI(location)));
				} catch (InvalidLocationException e) {
					// TODO: Prompt to the user whether to delete the location or not
					e.printStackTrace();
				}
			}
		}
	}
	
	private void loadActiveManager() throws NullLocationException, UninitializedLocationException, IOException {
		Preferences node = getPreferencesNode();
		String type = node.get(PREF_ACTIVE_MANAGER_TYPE, "");
		String location = node.get(PREF_ACTIVE_MANAGER_LOCATION, "");
		
		if (type != null && location != null) {
			Preferences typeNode = getPreferencesNode().node(type);
			activeManager = registeredManagers.get(typeNode.absolutePath() + location);
		} else {
			ICredentialsManager manager = getPriorityManager();
			if (manager != null) {
				try {
					setActiveManager(manager);
				} catch (BackingStoreException e) {
				}
			}
			activeManager = manager;
		}
		if (activeManager == null) {
			activeManager = new DummyCredentialsManager();
		}
		activeManager.loadLocation();
	}

	public CredentialsManagerType getManagerTypeFor(ICredentialsManager manager) {
		if (manager == null) {
			return null;
		}
		for (CredentialsManagerType type : registeredTypes) {
			if (type.getManagerClassName().equals(manager.getClass().getName())) {
				return type;
			}
		}
		return null;
	}
	
	public List<CredentialsManagerType> getManagerTypes() {
		return registeredTypes;
	}
	
	public Collection<ICredentialsManager> getManagers() {
		return registeredManagers.values();
	}
	
	public ICredentialsManager createManager(String typeId, URI uri) throws CoreException, InvalidLocationException, UnknownCredentialsManagerType  {
		ICredentialsManager manager;
		if (findType(typeId) != null) {
			CredentialsManagerType type = findType(typeId);
			manager = type.createCredentialsManager();
			manager.setLocation(uri);
		} else {
			throw new UnknownCredentialsManagerType(typeId);
		}
		return manager;
	}

	public void setActiveManager(String typeId, URI uri) throws BackingStoreException, NullLocationException, UninitializedLocationException, IOException {
		Preferences node = getPreferencesNode();
		
		if (typeId != null)
			node.put(PREF_ACTIVE_MANAGER_TYPE, typeId);
		else
			node.remove(PREF_ACTIVE_MANAGER_TYPE);

		if (uri != null)
			node.put(PREF_ACTIVE_MANAGER_LOCATION, uri.toString());
		else
			node.remove(PREF_ACTIVE_MANAGER_LOCATION);
			
		node.flush();
		loadActiveManager();
	}

	public void setActiveManager(ICredentialsManager manager) throws BackingStoreException, NullLocationException, UninitializedLocationException, IOException {
		CredentialsManagerType type = getManagerTypeFor(manager);
		String typeId = type != null ? type.getId() : null;
		URI location = manager != null ? manager.getLocation() : null;
		setActiveManager(typeId, location);
	}

	public ICredentialsManager getActiveManager() {
		return activeManager;
	}
	
	public void registerManager(ICredentialsManager manager) throws BackingStoreException {
		String typeId = getManagerTypeFor(manager).getId();
		URI uri = manager.getLocation();
		Preferences node = getPreferencesNode().node(typeId);
		String location = uri.toString();
		node.put(location, location);
		registeredManagers.put(node.absolutePath() + location, manager);
		node.flush();
	}
	
	public void registerManager(String typeId, URI uri) throws BackingStoreException, CoreException, InvalidLocationException, UnknownCredentialsManagerType  {
		registerManager(createManager(typeId, uri));
	}
	
	public void unregisterManager(ICredentialsManager manager) throws BackingStoreException {
		String typeId = getManagerTypeFor(manager).getId();
		unregisterManager(typeId, manager.getLocation());
	}
	
	public void unregisterManager(String typeId, URI uri) throws BackingStoreException {
		Preferences node = getPreferencesNode().node(typeId);
		String location = uri.toString();
		node.remove(location);
		registeredManagers.remove(node.absolutePath() + location);
		node.flush();
	}
	

	private Preferences getPreferencesNode() {
		return CredentialsManagerPlugin.getPreferences().node(PREF_NODE_REGISTERED_MANAGERS);
	}
	
	private CredentialsManagerType findType(String id) {
    	for (CredentialsManagerType type : registeredTypes) {
    		if (type.getId().equals(id)) {
    			return type;
    		}
    	}
		return null;
	}
	
	private ICredentialsManager getPriorityManager() {
		ICredentialsManager priorityManager = null;
		
		int lastPriority = Integer.MIN_VALUE;
		for (ICredentialsManager manager : registeredManagers.values()) {
			if (getManagerTypeFor(manager).getPriority() > lastPriority) {
				priorityManager = manager;
			}
		}
		
		return priorityManager;
	}
	
	class DummyCredentialsManager implements ICredentialsManager {

		private Organization organization = OmFactory.eINSTANCE.createOrganization();
		
		@Override
		public void setLocation(URI uri) throws InvalidLocationException {
		}

		@Override
		public URI getLocation() {
			return null;
		}

		@Override
		public void loadLocation() throws NullLocationException,
				UninitializedLocationException, IOException {
		}

		@Override
		public void initializeLocation() throws IOException {
		}

		@Override
		public boolean isValidCredential(UUID uuid, String password) {
			return false;
		}
		
		@Override
		public boolean isValidCredential(User user, String password) {
			return false;
		}
		
		@Override
		public String resolveActorName(UUID uuid) {
			if (uuid == null) {
				return null;
			} else {
				return uuid.toString();
			}
		}

		@Override
		public Organization getOrganization() {
			return organization;
		}

		@Override
		public Actor resolveActor(UUID uuid) {
			return null;
		}

		@Override
		public User resolveUser(String login) {
			return null;
		}

		@Override
		public boolean isIncluded(Actor actor1, Actor actor2) {
			return false;
		}
	}
}
