package es.upv.dsic.issi.dplfw.om.credentialsmanager.ui.preferences;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.filesystem.EFS;
import org.eclipse.core.filesystem.IFileStore;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.preference.PreferencePage;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.ComboViewer;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPreferencePage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.statushandlers.StatusManager;
import org.osgi.service.prefs.BackingStoreException;

import es.upv.dsic.issi.dplfw.om.credentialsmanager.CredentialsManagerPlugin;
import es.upv.dsic.issi.dplfw.om.credentialsmanager.CredentialsManagerType;
import es.upv.dsic.issi.dplfw.om.credentialsmanager.ICredentialsManager;
import es.upv.dsic.issi.dplfw.om.credentialsmanager.exceptions.InvalidLocationException;
import es.upv.dsic.issi.dplfw.om.credentialsmanager.exceptions.NullLocationException;
import es.upv.dsic.issi.dplfw.om.credentialsmanager.exceptions.UninitializedLocationException;
import es.upv.dsic.issi.dplfw.om.credentialsmanager.exceptions.UnknownCredentialsManagerType;
import es.upv.dsic.issi.dplfw.om.credentialsmanager.ui.CredentialsManagerUIPlugin;
import es.upv.dsic.issi.dplfw.om.credentialsmanager.ui.wizards.RegisterNewManagerWizard;
import es.upv.dsic.issi.dplfw.om.credentialsmanager.ui.wizards.RegisterNewManagerWizardValues;

public class CredentialsPreferencePage extends PreferencePage implements IWorkbenchPreferencePage {

	private final class CmsLabelProvider extends LabelProvider {
		@Override
		public String getText(Object element) {
			if (element instanceof ICredentialsManager) {
				ICredentialsManager manager = (ICredentialsManager) element;
				return String.format("%s (Location: %s)",
						CredentialsManagerPlugin.getDefault().getManagersRegistry().getManagerTypeFor(manager).getName(),
						manager.getLocation().toString());
			} else {
				return super.getText(element);
			}
		}

		@Override
		public Image getImage(Object element) {
			return CredentialsManagerUIPlugin.getDefault().getImageRegistry().get(
					CredentialsManagerUIPlugin.IMG_OBJ16_UNKNOWN_CM);
		}
	}

	private final class AddManagerAdapter extends SelectionAdapter {
		@Override
		public void widgetSelected(SelectionEvent e) {
			RegisterNewManagerWizard wizard = new RegisterNewManagerWizard();
			WizardDialog wizardDialog = new WizardDialog(getControl().getShell(), wizard);
			wizardDialog.create();
			wizardDialog.setBlockOnOpen(true);
			if (wizardDialog.open() == WizardDialog.OK) {
				RegisterNewManagerWizardValues values = wizard.getReturnValues();
				ICredentialsManager manager = null;
				try {
					manager = values.getType().createCredentialsManager();
				} catch (CoreException e1) {
					e1.printStackTrace();
				}

				try {
					manager.setLocation(values.getLocation());
					managers.add(manager);
				} catch (InvalidLocationException e1) {
					try {
						manager.initializeLocation();
						managers.add(manager);
					} catch (IOException e2) {
						e2.printStackTrace();
					}
				}
				managersViewer.refresh();
				comboViewer.refresh();
			}
		}
	}
	
	private final class RemoveManagerAdapter extends SelectionAdapter {
		@Override
		public void widgetSelected(SelectionEvent e) {
			IStructuredSelection selection = (IStructuredSelection) managersViewer.getSelection();
			managers.remove(selection.getFirstElement());
			managersViewer.refresh();
			comboViewer.refresh();
		}
	}


	private TableViewer managersViewer;
	private Button addManagerButton;
	private Button removeManagerButton;
	private Button editManagerButton;

	private List<ICredentialsManager> managers;
	private ComboViewer comboViewer;


	public CredentialsPreferencePage() {
		setDescription("Available credential managers");
		loadRegisteredManagers();
	}

	private void loadRegisteredManagers() {
		managers = new ArrayList<ICredentialsManager>();
		managers.addAll(CredentialsManagerPlugin.getDefault().getManagersRegistry().getManagers());
	}

	@Override
	protected Control createContents(Composite parent) {
		
		GridLayout topLayout = new GridLayout(2, false);
		topLayout.marginWidth = 0;
		topLayout.marginHeight = 0;

		GridData layoutData = new GridData();
		layoutData.grabExcessHorizontalSpace = true;
		layoutData.grabExcessVerticalSpace = true;
		layoutData.horizontalAlignment = SWT.FILL;
		layoutData.verticalAlignment = SWT.FILL;
		layoutData.minimumHeight = 200;
		layoutData.heightHint = 200;
		layoutData.widthHint = 500;
		layoutData.minimumWidth = 400;
		
		Composite topComposite = new Composite(parent, SWT.NONE);
		topComposite.setLayoutData(layoutData);
		topComposite.setLayout(topLayout);
		
		Label activeLabel = new Label(topComposite, SWT.NONE);
		activeLabel.setText("&Active Manager:");
		activeLabel.setLayoutData(new GridData(SWT.FILL, SWT.BEGINNING, true, false, 2, 1));
		
		comboViewer = new ComboViewer(topComposite, SWT.DROP_DOWN | SWT.BORDER | SWT.READ_ONLY);
		GridData comboLayoutData = new GridData(SWT.FILL, SWT.BEGINNING, true, false, 2, 1);
		comboLayoutData.widthHint = 400;
		comboViewer.getControl().setLayoutData(comboLayoutData);
		comboViewer.setLabelProvider(new CmsLabelProvider());
		comboViewer.setContentProvider(new ArrayContentProvider());
		comboViewer.setInput(managers);
		
		comboViewer.setSelection(new StructuredSelection(
				CredentialsManagerPlugin.getDefault().getManagersRegistry().getActiveManager()));
				
		Label availableManagersLabel = new Label(topComposite, SWT.NONE);
		availableManagersLabel.setText("Available Managers:");
		availableManagersLabel.setLayoutData(new GridData(SWT.FILL, SWT.BEGINNING, true, false, 2, 1));

		
		managersViewer = new TableViewer(topComposite, SWT.FULL_SELECTION | SWT.BORDER | SWT.SINGLE);
		managersViewer.setContentProvider(new ArrayContentProvider());
		managersViewer.setLabelProvider(new CmsLabelProvider());
		managersViewer.setInput(managers);
		managersViewer.addSelectionChangedListener(new ISelectionChangedListener() {
			@Override
			public void selectionChanged(SelectionChangedEvent event) {
				removeManagerButton.setEnabled(!event.getSelection().isEmpty());
				editManagerButton.setEnabled(!event.getSelection().isEmpty());
			}
		});
		GridData viewerLayoutData = new GridData(SWT.FILL, SWT.FILL, true, true);
		viewerLayoutData.widthHint = 400;
		managersViewer.getControl().setLayoutData(viewerLayoutData);
		
		
		Composite buttonsComposite = new Composite(topComposite, SWT.NONE);
		GridData gridData = new GridData(100, SWT.DEFAULT);
		gridData.grabExcessVerticalSpace = false;
		gridData.verticalAlignment = SWT.BEGINNING;
		buttonsComposite.setLayoutData(gridData);
		FillLayout fillLayout = new FillLayout(SWT.VERTICAL);
		fillLayout.spacing = 5;
		buttonsComposite.setLayout(fillLayout);
		
		addManagerButton = new Button(buttonsComposite, SWT.PUSH);
		addManagerButton.setText("&Add...");
		addManagerButton.addSelectionListener(new AddManagerAdapter());
		
		removeManagerButton = new Button(buttonsComposite, SWT.PUSH);
		removeManagerButton.setText("&Remove");
		removeManagerButton.addSelectionListener(new RemoveManagerAdapter());
		removeManagerButton.setEnabled(false);

		Label separator = new Label(buttonsComposite, SWT.NONE);
		
		editManagerButton = new Button(buttonsComposite, SWT.PUSH);
		editManagerButton.setText("&Open in Editor");
		editManagerButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				
				IStructuredSelection selection = (IStructuredSelection) managersViewer.getSelection();
				ICredentialsManager manager = (ICredentialsManager) selection.getFirstElement();
				
				URI uri = manager.getLocation();
				IWorkbenchPage page = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage();
				
				IFileStore fileStore = EFS.getLocalFileSystem().getStore(new Path(new File(uri).toString()));
				if (!fileStore.fetchInfo().isDirectory() && fileStore.fetchInfo().exists()) {
			        try {
						IDE.openEditorOnFileStore(page, fileStore);
					} catch (PartInitException e1) {
						logException(e1);
					}
				}
			}
		});
		editManagerButton.setEnabled(false);

		
		return topComposite;
	}


	@Override
	public void init(IWorkbench workbench) {
	}
	
	@Override
	protected void performDefaults() {
		super.performDefaults();
		loadRegisteredManagers();
	}
	
	@Override
	public boolean performOk() {
		boolean returnValue = true;
		ArrayList<ICredentialsManager> currentManagers = new ArrayList<ICredentialsManager>();
		currentManagers.addAll(CredentialsManagerPlugin.getDefault().getManagersRegistry().getManagers());

		// 1st. step: unregister deleted managers
		for (ICredentialsManager currentManager : currentManagers) {
			if (!managers.contains(currentManager)) {
				CredentialsManagerType type = CredentialsManagerPlugin.getDefault().
						getManagersRegistry().getManagerTypeFor(currentManager);
				
				try {
					CredentialsManagerPlugin.getDefault().getManagersRegistry().unregisterManager(
							type.getId(), currentManager.getLocation());
				} catch (BackingStoreException e) {
					logException(e);
					returnValue = false;
				}
			}
		}

		// 2nd step: register new managers
		for (ICredentialsManager manager : managers) {
			CredentialsManagerType type = CredentialsManagerPlugin.getDefault().
					getManagersRegistry().getManagerTypeFor(manager);
			
			try {
				CredentialsManagerPlugin.getDefault().getManagersRegistry().registerManager(
						type.getId(), manager.getLocation());
			} catch (CoreException e) {
				logException(e);
				returnValue = false;
			} catch (InvalidLocationException e) {
				logException(e);
				returnValue = false;
			} catch (UnknownCredentialsManagerType e) {
				logException(e);
				returnValue = false;
			} catch (BackingStoreException e) {
				logException(e);
				returnValue = false;
			}
		}
		// 3rd. Save selected active manager
		// This step must be the last, since the previous steps update the available managers
		IStructuredSelection activeManagerSelection = (IStructuredSelection) comboViewer.getSelection();
		try {
			CredentialsManagerPlugin.getDefault().getManagersRegistry().setActiveManager(
					(ICredentialsManager) activeManagerSelection.getFirstElement());
		} catch (BackingStoreException e) {
			logException(e);
			returnValue = false;
		} catch (NullLocationException e) {
			logException(e);
			returnValue = false;
		} catch (UninitializedLocationException e) {
			logException(e);
			returnValue = false;
		} catch (IOException e) {
			logException(e);
			returnValue = false;
		}
				
		return returnValue;
	}
	
	void logException(Exception e) {
		StatusManager.getManager().handle(
				new Status(Status.ERROR, CredentialsManagerUIPlugin.PLUGIN_ID, e.getLocalizedMessage(), e),
				StatusManager.BLOCK | StatusManager.LOG);
	}
}