package es.upv.dsic.issi.dplfw.infoelement.singleeditor.editor.image;

import java.io.ByteArrayInputStream;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Spinner;
import org.eclipse.ui.forms.widgets.SharedScrolledComposite;
import org.eclipse.ui.forms.widgets.TableWrapData;
import org.eclipse.ui.forms.widgets.TableWrapLayout;

import es.upv.dsic.issi.dplfw.core.ui.util.SWTUtil;


public class SizedImageEditControl extends Composite {

	private final class ResizeListener implements Listener {
		Runnable runnable = new Runnable()  {
			@Override
			public void run() {
				if (!isDisposed()) {
					setRedraw(false);
					refresh();
					resize = false;
					setRedraw(true);
				}
			}
		};

		private volatile boolean resize = false;

		@Override
		public void handleEvent(Event event) {
			if (!resize) {
				if (imageLabel != null && imageLabel.getImage() != null) {
					int height = imageLabel.getSize().y + DEFAULT_COMPOSITE_MARGIN * 2 + 4;
					imageLabel.getImage().dispose();
					imageLabel.setImage(null);
					TableWrapData tableWrapData = new TableWrapData(TableWrapData.FILL_GRAB);
					tableWrapData.heightHint = height;
					imageComposite.setLayoutData(tableWrapData);
					imageComposite.setBackground(Display.getDefault().getSystemColor(SWT.COLOR_WIDGET_BACKGROUND));
					imageLabel.setBackground(Display.getDefault().getSystemColor(SWT.COLOR_WIDGET_BACKGROUND));
				}
				getShell().getDisplay().timerExec(250, runnable);
			}
			resize = true;
		}
	}

	private Spinner heightSpinner;
	private Spinner widthSpinner;
	private Label imageLabel;
	private Button loadButton;
	private Button resetButton;
	private Image image;
	private Composite imageComposite;
	private Composite modifiersComposite;
	private boolean needsReload = false; // Used when reloading with an empty image
	private int lastWidth;
	
	ResizeListener resizeListener = new ResizeListener();
	private int widthDifference;
	private static int DEFAULT_COMPOSITE_MARGIN = 5;
	private Composite editComposite;
	private Button keepAspectRatioButton;
	
	
	public SizedImageEditControl(Composite parent, int style) {
		super(parent, style);
		enableScrollsInParents();
		
		setLayout(new TableWrapLayout());

		editComposite = new Composite(this, SWT.NONE);
		TableWrapLayout tableWrapLayout = new TableWrapLayout();
		tableWrapLayout.numColumns = 2;
		tableWrapLayout.makeColumnsEqualWidth = false;
		editComposite.setLayout(tableWrapLayout);
		editComposite.setLayoutData(new TableWrapData(TableWrapData.FILL_GRAB));
		
		{ // Image composite. Fill horizontal.
			imageComposite = new Composite(editComposite, SWT.BORDER | SWT.FLAT);
			imageComposite.setLayoutData(new TableWrapData(TableWrapData.FILL_GRAB));
			
			TableWrapLayout imageCompositeLayout = new TableWrapLayout();
			imageCompositeLayout.topMargin = 0;
			
			imageComposite.setLayout(imageCompositeLayout);
			
			// dummy composite to force that the image is centered
			Composite dummy = new Composite(imageComposite, SWT.NONE);
			dummy.setLayoutData(new TableWrapData(TableWrapData.FILL_GRAB));
			GridLayout layout = new GridLayout();
			layout.marginWidth = 0;
			layout.marginHeight = 0;
			layout.verticalSpacing = 0;
			dummy.setLayout(layout);
			
			// Dummy composite to guarantee a minimum width
			Composite dummyDummyComposite = new Composite(dummy, SWT.NONE);
			GridData data = new GridData();
			data.widthHint = 1;
			data.heightHint = 0;
			dummyDummyComposite.setLayoutData(data);
			
			imageLabel = new Label(imageComposite, SWT.NONE);
			imageLabel.setLayoutData(new TableWrapData(TableWrapData.CENTER));
			
		}
		{ // Details composite. Row layout.
			modifiersComposite = new Composite(editComposite, SWT.NONE);
			modifiersComposite.setLayoutData(new TableWrapData());
			modifiersComposite.setLayout(new TableWrapLayout());
	
			
			loadButton = new Button(modifiersComposite, SWT.PUSH);
			loadButton.setText("Load image...");
			loadButton.setLayoutData(new TableWrapData(TableWrapData.FILL_GRAB));
			
			resetButton = new Button(modifiersComposite, SWT.PUSH);
			resetButton.setText("Delete image");
			resetButton.setLayoutData(new TableWrapData(TableWrapData.FILL_GRAB));
	
			new Label(modifiersComposite, SWT.NONE);
			Label prefLabel = new Label(modifiersComposite, SWT.NONE);
			prefLabel.setText("Preferred image size:");
			
			{
				Composite sizesComposite = new Composite(modifiersComposite, SWT.NONE);
				TableWrapLayout sizesCompositeLayout = new TableWrapLayout();
				sizesCompositeLayout.numColumns = 2;
				sizesCompositeLayout.makeColumnsEqualWidth = false;
				sizesCompositeLayout.topMargin = 5;
				sizesCompositeLayout.bottomMargin = 5;
				sizesCompositeLayout.leftMargin = 10;
				sizesCompositeLayout.rightMargin = 10;
				sizesCompositeLayout.horizontalSpacing = 10;
				sizesComposite.setLayout(sizesCompositeLayout);
				
				Label widthLabel = new Label(sizesComposite, SWT.NONE);
				widthLabel.setText("Width:");
				widthSpinner = new Spinner(sizesComposite, SWT.BORDER);
				widthSpinner.setLayoutData(new TableWrapData(TableWrapData.FILL_GRAB));
				widthSpinner.setMinimum(-1);
				widthSpinner.setMaximum((int) Math.pow(2, 13));
				widthSpinner.setToolTipText("Preferred width to display the image in pixels. Use '-1' to set the value to the image's actual width.");
		
				Label heightLabel = new Label(sizesComposite, SWT.NONE);
				heightLabel.setText("Height:");
				heightSpinner = new Spinner(sizesComposite, SWT.BORDER);
				heightSpinner.setLayoutData(new TableWrapData(TableWrapData.FILL_GRAB));
				heightSpinner.setMinimum(-1);
				heightSpinner.setMaximum((int) Math.pow(2, 13));
				heightSpinner.setToolTipText("Preferred height to display the image in pixels. Use '-1' to set the value to the image's actual height.");

				
				keepAspectRatioButton = new Button(sizesComposite, SWT.CHECK);
				keepAspectRatioButton.setText("Keep aspect ratio");
				keepAspectRatioButton.setSelection(true);
				keepAspectRatioButton.setLayoutData(new TableWrapData(TableWrapData.FILL_GRAB, TableWrapData.TOP, 1, 2));
				
				heightSpinner.addSelectionListener(new SelectionAdapter() {
					@Override
					public void widgetDefaultSelected(SelectionEvent e) {
						updateWidthSpinnerFromHeight();
					}
					@Override
					public void widgetSelected(SelectionEvent e) {
						updateWidthSpinnerFromHeight();
					}
				});
				
				widthSpinner.addSelectionListener(new SelectionAdapter() {
					@Override
					public void widgetDefaultSelected(SelectionEvent e) {
						updateHeightSpinnerFromWidth();
					}
					@Override
					public void widgetSelected(SelectionEvent e) {
						updateHeightSpinnerFromWidth();
					}
				});
			}
		}
		
		this.addListener(SWT.Resize, resizeListener);
		
		layout(true, true);
		widthDifference = Math.max(modifiersComposite.getSize().x + DEFAULT_COMPOSITE_MARGIN * 8 + 2, 1);

		createDisposeListener();
	}
	
	private void updateHeightSpinnerFromWidth() {
		if (widthSpinner.getSelection() < 0) {
			widthSpinner.setSelection(image.getImageData().width);
		} else if (keepAspectRatioButton.getSelection()) {
			if (image != null) {
				float ratio = (float) image.getImageData().height / image.getImageData().width;
				int newValue = Math.round((float) widthSpinner.getSelection() * ratio);
				if (newValue != heightSpinner.getSelection()) {
					heightSpinner.setSelection(newValue);
				}
			}
		}
	}
	
	private void updateWidthSpinnerFromHeight() {
		if (heightSpinner.getSelection() < 0) {
			heightSpinner.setSelection(image.getImageData().height);
		} else if (keepAspectRatioButton.getSelection()) {
			if (image != null) {
				float ratio = (float) image.getImageData().width / image.getImageData().height;
				int newValue = Math.round((float) heightSpinner.getSelection() * ratio);
				if (newValue != widthSpinner.getSelection()) {
					widthSpinner.setSelection(newValue);
				}
			}
		}
	}
	/**
	 * Enables {@link SharedScrolledComposite#setAlwaysShowScrollBars(boolean)} in parent
	 * {@link SharedScrolledComposite}s to avoid loops when resizing
	 */
	private void enableScrollsInParents() {
		Composite parent = getParent();
		while (parent != null) {
			if (parent instanceof SharedScrolledComposite) {
				((SharedScrolledComposite) parent).setAlwaysShowScrollBars(true);
			}
			parent = parent.getParent();
		}
		
	}

	public Button getResetButton() {
		return resetButton;
	}
	
	public Button getLoadButton() {
		return loadButton;
	}
	
	public Button getKeepAspectRatioButton() {
		return keepAspectRatioButton;
	}

	public synchronized void setImageBytes(byte[] bytes) {
		if (this.image != null)
			this.image.dispose();
		this.image = new Image(getDisplay(), new ByteArrayInputStream(bytes));
		refresh();
	}

	private void internalImageUpdate() {
		if (image != null) {
			int maxWidth = Math.max(getSize().x - widthDifference, 1);
			if (needsReload || lastWidth != maxWidth) {
				needsReload = false; 
				if (imageLabel.getImage() != null) {
					imageLabel.getImage().dispose();
				}
				Image newImage;
				if (image.getBounds().width > maxWidth) {
					newImage = SWTUtil.createResizedImage(image, Math.max(maxWidth, 1));
				} else {
					newImage = new Image(getShell().getDisplay(), image, SWT.IMAGE_COPY);
				}
				imageLabel.setImage(newImage);
				imageComposite.setLayoutData(new TableWrapData(TableWrapData.FILL_GRAB));
				imageComposite.setBackground(null);
				lastWidth = newImage.getBounds().width;
				setSize(computeSize(getSize().x, SWT.DEFAULT));
				layout(true, true);
			}
		}
	}
	
	public Spinner getWidthSpinner() {
		return widthSpinner;
	}
	
	public Spinner getHeightSpinner() {
		return heightSpinner;
	}
	
	public void refresh() {
		internalImageUpdate();
		needsReload = true;
		notifyListeners(SWT.Resize, new Event());
	}
	private void createDisposeListener() {
		this.addDisposeListener(new DisposeListener() {
			@Override
			public void widgetDisposed(DisposeEvent e) {
				if (image != null) {
					image.dispose();
				}
				if (imageLabel != null && imageLabel.getImage() != null) {
					imageLabel.getImage().dispose();
				}
			}
		});
	}
}
