package es.upv.dsic.issi.dplfw.dfmconf.constraints;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.validation.AbstractModelConstraint;
import org.eclipse.emf.validation.EMFEventType;
import org.eclipse.emf.validation.IValidationContext;
import org.eclipse.emf.validation.model.ConstraintStatus;
import org.eclipse.emf.validation.service.ConstraintRegistry;
import org.eclipse.emf.validation.service.IConstraintDescriptor;

import es.upv.dsic.issi.dplfw.dfmconf.DocumentFeatureSelection;
import es.upv.dsic.issi.dplfw.dfmconf.RequiresSelection;


/**
 * Checks if a selection change of a feature affects to a requires 
 * relationship from a parent (isRequiredBy) feature 
 * 
 * The check is done from the point of view of the feature that requires 
 * the one that has changed, since it responds to a RequiresSelectionChangeEvent
 * 
 * @author agomez
 *
 */
public class RequiresConstraint extends AbstractModelConstraint {

	public IStatus validate(IValidationContext context) {
		
		EObject eObject = context.getTarget();
		EMFEventType eventType = context.getEventType();
	
		List<IStatus> statuses = new ArrayList<IStatus>();
		
		// In the case of batch mode.
		if (eventType == EMFEventType.NULL) {
			if (eObject instanceof DocumentFeatureSelection) {
				DocumentFeatureSelection documentFeatureSelection = (DocumentFeatureSelection) eObject;
				statuses.addAll(validateDocumentFeatureSelection(context, documentFeatureSelection));
			}
		// Live mode
		} else {
			DocumentFeatureSelection documentFeatureSelection = (DocumentFeatureSelection) context.getTarget();
			statuses.addAll(validateDocumentFeatureSelection(context, documentFeatureSelection));
		}
		if (statuses != null && !statuses.isEmpty()) {
			return ConstraintStatus.createMultiStatus(context, statuses);
		}
		return context.createSuccessStatus();
	}


	/** 
	 * This method returns a list of lists of document features
	 * The top level list represents an AND 
	 * The inner lists represent an OR
	 * 
	 * @param documentFeatureSelection
	 * @return
	 */
	protected List<List<DocumentFeatureSelection>> getRequirements(DocumentFeatureSelection documentFeatureSelection) {
		List<List<DocumentFeatureSelection>> requirements = new ArrayList<List<DocumentFeatureSelection>>();
		
		// Collect requirements
		for (RequiresSelection requiresSelection : documentFeatureSelection.getRequires()) {
			List<DocumentFeatureSelection> candidates = new ArrayList<DocumentFeatureSelection>();
			candidates.addAll(requiresSelection.getCandidates());
			requirements.add(candidates);
		}
		return requirements;
	}
	

	/**
	 * Checks if the requirements are met. At least one of the candidates for all the requirements must be selected.
	 * @param context
	 * @param documentFeatureSelection
	 * @return
	 */
	protected List<IStatus> validateDocumentFeatureSelection(IValidationContext context, DocumentFeatureSelection documentFeatureSelection) {
		List<IStatus> statuses = new ArrayList<IStatus>();

		List<List<DocumentFeatureSelection>> requirements = getRequirements(documentFeatureSelection);
		List<List<DocumentFeatureSelection>> unmetReqs = new ArrayList<List<DocumentFeatureSelection>> ();
		
		if (documentFeatureSelection.getSelected() == Boolean.TRUE && !requirements.isEmpty()) {
			boolean allSelected = true;
			// Iterate over the AND Requirements
			for (List<DocumentFeatureSelection> candidates : requirements) {
				boolean selectedCandidate = false;
				// Iterate over the OR Requirements (only one selection is required to continue)
				for (DocumentFeatureSelection selection : candidates) {
					if (selection.getSelected() == Boolean.TRUE) {
						selectedCandidate = true;
						break;
					}
				}
				if (!selectedCandidate) {
					allSelected = false;
					unmetReqs.add(candidates);
				}
			}
			if (!allSelected) {
				statuses.add(buildErrorStatus(context, documentFeatureSelection, unmetReqs));
			}
		}
		return statuses;
	}

	protected IStatus buildErrorStatus(IValidationContext context,
			DocumentFeatureSelection featureSelection, List<List<DocumentFeatureSelection>> requirements) {
		
		IConstraintDescriptor desc = ConstraintRegistry.getInstance().getDescriptor(
				context.getCurrentConstraintId());
		
		List<IStatus> statuses = new ArrayList<IStatus>();
		
		for (List<DocumentFeatureSelection> candidates : requirements) {
			StringBuilder builder = new StringBuilder();
			String separator = " or ";
			for (DocumentFeatureSelection selection : candidates) {
				builder.append("\"");
				builder.append(selection.getDocumentFeature().getVisibleName());
				builder.append("\"");
				builder.append(separator);
			}
			builder.setLength(builder.length() - separator.length());
			
			statuses.add(ConstraintStatus.createStatus(context,
					featureSelection,
					candidates,
					IStatus.WARNING,
					desc.getStatusCode(),
					"{0} {1} required by \"{2}\"",
					builder.toString(),
					candidates.size() > 1 ? "are" : "is",
					featureSelection.getDocumentFeature().getVisibleName()
			));
		}
		
		statuses.add(ConstraintStatus.createStatus(context, featureSelection,
				null,
				"Requirements of feature\"{0}\" are not met",
				featureSelection.getDocumentFeature().getVisibleName()));
		
		IStatus resultStatus = ConstraintStatus.createMultiStatus(context, statuses);
		
		return resultStatus;
	}
}
