package es.upv.dsic.issi.dplfw.dfmconf.constraints;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.validation.AbstractModelConstraint;
import org.eclipse.emf.validation.EMFEventType;
import org.eclipse.emf.validation.IValidationContext;
import org.eclipse.emf.validation.model.ConstraintStatus;
import org.eclipse.emf.validation.service.ConstraintRegistry;
import org.eclipse.emf.validation.service.IConstraintDescriptor;

import es.upv.dsic.issi.dplfw.dfmconf.DocumentFeatureSelection;
import es.upv.dsic.issi.dplfw.dfmconf.ExcludesSelection;

public class ExcludesConstraint extends AbstractModelConstraint {
	public IStatus validate(IValidationContext context) {
		
		EObject eObject = context.getTarget();
		EMFEventType eventType = context.getEventType();
	
		List<IStatus> statuses = new ArrayList<IStatus>();
		
		// In the case of batch mode.
		if (eventType == EMFEventType.NULL) {
			if (eObject instanceof DocumentFeatureSelection) {
				DocumentFeatureSelection documentFeatureSelection = (DocumentFeatureSelection) eObject;
				statuses.addAll(validateDocumentFeatureSelection(context, documentFeatureSelection));
			}
		// Live mode
		} else {
			DocumentFeatureSelection documentFeatureSelection = (DocumentFeatureSelection) context.getTarget();
			statuses.addAll(validateDocumentFeatureSelection(context, documentFeatureSelection));
		}
		if (statuses != null && !statuses.isEmpty()) {
			return ConstraintStatus.createMultiStatus(context, statuses);
		}
		return context.createSuccessStatus();
	}


	/** 
	 * This method returns a list of lists of document features
	 * The top level list represents an AND 
	 * The inner lists represent an OR
	 * 
	 * @param documentFeatureSelection
	 * @return
	 */
	protected List<List<DocumentFeatureSelection>> getExclusions(DocumentFeatureSelection documentFeatureSelection) {
		List<List<DocumentFeatureSelection>> exclusions = new ArrayList<List<DocumentFeatureSelection>>();
		
		// Collect exclusions
		for (ExcludesSelection excludesSelection : documentFeatureSelection.getExcludes()) {
			List<DocumentFeatureSelection> candidates = new ArrayList<DocumentFeatureSelection>();
			candidates.addAll(excludesSelection.getCandidates());
			exclusions.add(candidates);
		}
		return exclusions;
	}
	

	/**
	 * Checks if the exclusions are met. 
	 * @param context
	 * @param documentFeatureSelection
	 * @return
	 */
	protected List<IStatus> validateDocumentFeatureSelection(IValidationContext context, DocumentFeatureSelection documentFeatureSelection) {
		List<IStatus> statuses = new ArrayList<IStatus>();

		List<List<DocumentFeatureSelection>> exclusions = getExclusions(documentFeatureSelection);
		List<List<DocumentFeatureSelection>> unmetExclusions = new ArrayList<List<DocumentFeatureSelection>> ();
		
		if (documentFeatureSelection.getSelected() == Boolean.TRUE && !exclusions.isEmpty()) {
			boolean allUnSelected = true;
			// Iterate over the AND Requirements
			for (List<DocumentFeatureSelection> candidates : exclusions) {
				boolean selectedCandidate = false;
				// Iterate over the OR Requirements (at least one exclusion must be unselected)
				List<DocumentFeatureSelection> unmetCandidates = new ArrayList<DocumentFeatureSelection>();
				for (DocumentFeatureSelection selection : candidates) {
					if (selection.getSelected() == Boolean.TRUE) {
						selectedCandidate = true;
						unmetCandidates.add(selection);
					}
				}
				if (selectedCandidate) {
					allUnSelected = false;
					unmetExclusions.add(unmetCandidates);
				}
			}
			if (!allUnSelected) {
				statuses.add(buildErrorStatus(context, documentFeatureSelection, unmetExclusions));
			}
		}
		return statuses;
	}

	protected IStatus buildErrorStatus(IValidationContext context,
			DocumentFeatureSelection featureSelection, List<List<DocumentFeatureSelection>> requirements) {
		
		IConstraintDescriptor desc = ConstraintRegistry.getInstance().getDescriptor(
				context.getCurrentConstraintId());
		
		List<IStatus> statuses = new ArrayList<IStatus>();
		
		for (List<DocumentFeatureSelection> candidates : requirements) {
			StringBuilder builder = new StringBuilder();
			String separator = " or ";
			for (DocumentFeatureSelection selection : candidates) {
				builder.append("\"");
				builder.append(selection.getDocumentFeature().getVisibleName());
				builder.append("\"");
				builder.append(separator);
			}
			builder.setLength(builder.length() - separator.length());
			
			statuses.add(ConstraintStatus.createStatus(context,
					featureSelection,
					candidates,
					IStatus.WARNING,
					desc.getStatusCode(),
					"{0} {1} excluded by \"{2}\"",
					builder.toString(),
					candidates.size() > 1 ? "are" : "is",
					featureSelection.getDocumentFeature().getVisibleName()
			));
		}
		
		statuses.add(ConstraintStatus.createStatus(context, featureSelection,
				null,
				"Exclusions of feature\"{0}\" are not met",
				featureSelection.getDocumentFeature().getVisibleName()));
		
		IStatus resultStatus = ConstraintStatus.createMultiStatus(context, statuses);
		
		return resultStatus;
	}
}
