package es.upv.dsic.issi.dplfw.core.util;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.UUID;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.eclipse.emf.cdo.session.CDOSession;
import org.eclipse.emf.cdo.view.CDOView;
import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.w3c.dom.Document;
import org.w3c.dom.DocumentType;
import org.w3c.dom.Element;

import es.upv.dsic.issi.dplfw.core.DplfwPlugin;
import es.upv.dsic.issi.dplfw.dfm.ContentDocumentFeature;
import es.upv.dsic.issi.dplfw.dfm.DocumentFeature;
import es.upv.dsic.issi.dplfw.dfmconf.AttributeConfiguration;
import es.upv.dsic.issi.dplfw.dfmconf.DocumentFeatureModelConfiguration;
import es.upv.dsic.issi.dplfw.dfmconf.DocumentFeatureSelection;
import es.upv.dsic.issi.dplfw.dfmconf.Reference;
import es.upv.dsic.issi.dplfw.dfmconf.VariableAttributeConfiguration;
import es.upv.dsic.issi.dplfw.infoelements.DitaRepresentable;
import es.upv.dsic.issi.dplfw.infoelements.InfoElement;
import es.upv.dsic.issi.dplfw.infoelements.TextIE;
import es.upv.dsic.issi.dplfw.infoelements.VariableIEContents;
import es.upv.dsic.issi.dplfw.om.credentialsmanager.CredentialsManagerPlugin;
import es.upv.dsic.issi.dplfw.repomanager.IRepositoryManager;
import es.upv.dsic.issi.dplfw.repomanager.RepositoryLocation;
import es.upv.dsic.issi.dplfw.repomanager.UnknownRepositoryException;
import es.upv.dsic.issi.dplfw.wfm.Activity;
import es.upv.dsic.issi.dplfw.wfm.Actor;
import es.upv.dsic.issi.dplfw.wfm.End;
import es.upv.dsic.issi.dplfw.wfm.FlowNode;
import es.upv.dsic.issi.dplfw.wfm.IntermediateNode;
import es.upv.dsic.issi.dplfw.wfm.Process;
import es.upv.dsic.issi.dplfw.wfm.Source;
import es.upv.dsic.issi.dplfw.wfm.Start;
import es.upv.dsic.issi.dplfw.wfm.Subprocess;
import es.upv.dsic.issi.dplfw.wfm.Task;
import es.upv.dsic.issi.dplfw.wfm.WfmFactory;
import nu.xom.Serializer;
import nu.xom.converters.DOMConverter;

public class TransformerUtil {

	private static final String VARIABLES_DITAMAP_FILENAME = "variables.ditamap";
	private static final String MAIN_DITAMAP_FILENAME = "main.ditamap";

	/**
	 * Transforms a {@link DocumentFeatureModelConfiguration} into a
	 * {@link Process}. It is noteworthy to remark that {@link InfoElement}s
	 * retrieval is done during this process. Thus, <b>it is important that the
	 * {@link RepositoryLocation} of the linked {@link InfoElement}s is properly
	 * registered in the {@link IRepositoryManager#INSTANCE}</b>.
	 * 
	 * @param dfmc
	 * @return a {@link Process}
	 */
	public static Process transformDfmconfToProcess(DocumentFeatureModelConfiguration dfmc) {
		EcoreUtil.resolveAll(dfmc);
		HashMap<String, Subprocess> subprocessMap = new HashMap<String, Subprocess>();
		HashMap<String, Task> taskMap = new HashMap<String, Task>();
		Process process = tConfig2Process(dfmc);
		tContentDocumentFeatureSelection2Activity(dfmc, process, subprocessMap, taskMap);
		tProcessStartEnd(process);
		tLinkFlowNodes(dfmc, process, subprocessMap, taskMap);
		return process;
	}
	
	private static Process tConfig2Process(DocumentFeatureModelConfiguration dfmc) {
		Process process = WfmFactory.eINSTANCE.createProcess();
		process.setName(dfmc.getDocumentFeatureModel().getName());
		return process;
	}
	
	
	private static void tContentDocumentFeatureSelection2Activity(DocumentFeatureModelConfiguration dfmc, Process process, HashMap<String, Subprocess> subprocessMap, HashMap<String, Task> taskMap) {
		for (DocumentFeatureSelection dfs : collectChildren(dfmc, DocumentFeatureSelection.class)) {		
			if (dfs.getSelected() == Boolean.TRUE && dfs.getDocumentFeature() instanceof ContentDocumentFeature) {
				ContentDocumentFeature cdf = (ContentDocumentFeature) dfs.getDocumentFeature();
				Activity activity;
				if (dfs.getChildrenSelection().isEmpty()) {
					activity = getTask(cdf, taskMap);
				} else {
					activity = getSubprocess(cdf, subprocessMap);
				}
				Process currentProcess;
				if (dfs.getParentSelection() != null) {
					currentProcess = getSubprocess(dfs.getParentSelection().getDocumentFeature(), subprocessMap);
				} else {
					currentProcess = process;
				}
				
				currentProcess.getNodes().add((FlowNode) activity);
				
				try {
					if(dfs.getCriterionAttributesConfiguration() != null && dfs.getCriterionAttributesConfiguration().getInfoElementURICandidate() != null){
						activity.setInfoElement(loadInfoElement(dfs.getCriterionAttributesConfiguration().getInfoElementURICandidate()));
					} else if (cdf.getInfoElementURI() != null) {
						activity.setInfoElement(loadInfoElement(cdf.getInfoElementURI()));
						// We recreate the ID to prevent notification of errors
						//activity.getInfoElement().createUUID();
					}
				} catch (UnknownRepositoryException e) {
					DplfwPlugin.log(e);
				}

				if (cdf.getResponsible() != null) {
					activity.setResponsible(createActor(cdf.getResponsible()));
				}
				
				if (cdf.getEditors().isEmpty()) {
					activity.setAproved(true);
				} else {
					for (UUID editor : cdf.getEditors()) {
						activity.getEditors().add(createActor(editor));
					}
				}

				for (UUID reader : cdf.getReaders()) {
					activity.getReaders().add(createActor(reader));
				}
				
				activity.setDocumentFeatureSelection(dfs);
				
			}
		}
	}
	
	private static Actor createActor(UUID uuid) {
		Actor actor = WfmFactory.eINSTANCE.createActor();
		actor.setUuid(uuid);
		actor.setName(CredentialsManagerPlugin.getDefault().getManagersRegistry().getActiveManager().resolveActorName(uuid));
		return actor;
	}
	
	private static void tProcessStartEnd(Process process) {
		collectChildren(process, Process.class).forEach(p -> {
			Start start = WfmFactory.eINSTANCE.createStart();
			End end = WfmFactory.eINSTANCE.createEnd();
			p.setStart(start);
			p.setEnd(end);
		});
	}
	
	private static void tLinkFlowNodes(DocumentFeatureModelConfiguration dfmc, Process process, HashMap<String, Subprocess> subprocessMap, HashMap<String, Task> taskMap) {
		EList<DocumentFeatureSelection> topDocumentFeaturesSelection = new BasicEList<DocumentFeatureSelection>();
		for (DocumentFeatureSelection sel : dfmc.getTopFeaturesSelection()) {
			if (sel.getDocumentFeature() instanceof ContentDocumentFeature)
				topDocumentFeaturesSelection.add(sel);
		}
		buildChain(process, topDocumentFeaturesSelection, subprocessMap, taskMap);
		for (DocumentFeatureSelection dfs : collectChildren(dfmc, DocumentFeatureSelection.class)) {
			if (dfs.getSelected() == Boolean.TRUE && dfs.getDocumentFeature() instanceof ContentDocumentFeature) {
				Subprocess subprocess = getSubprocess(dfs.getDocumentFeature(), subprocessMap);
				EList<DocumentFeatureSelection> childrenSelection = dfs.getChildrenSelection();
				buildChain(subprocess, childrenSelection, subprocessMap, taskMap);
			}
		}
	}

	private static void buildChain(Process process, EList<DocumentFeatureSelection> childrenSelection, HashMap<String, Subprocess> subprocessMap, HashMap<String, Task> taskMap) {
		Source source = (Source) process.getStart();
		for (DocumentFeatureSelection childDfs : childrenSelection) {
			if (childDfs.getSelected() == Boolean.TRUE) {
				IntermediateNode node;
				if (childDfs.getChildrenSelection().isEmpty()) {
					node = getTask(childDfs.getDocumentFeature(), taskMap);
				} else { 
					node = getSubprocess(childDfs.getDocumentFeature(), subprocessMap);
				}
				source.getTo().add(node);
				source = node;
			}
		}
		if (source != null) {
			source.getTo().add(process.getEnd());
		}
	}
	
	private static Task getTask(DocumentFeature df, HashMap<String, Task> taskMap) {
		if (!taskMap.containsKey(df.getIdName())) {
			Task task = WfmFactory.eINSTANCE.createTask();
			task.setName(df.getVisibleName());
			taskMap.put(df.getIdName(), task);
		}
		return taskMap.get(df.getIdName());
	}
	
	private static Subprocess getSubprocess(DocumentFeature df, HashMap<String, Subprocess> subprocessMap) {
		if (!subprocessMap.containsKey(df.getIdName())) {
			Subprocess subprocess = WfmFactory.eINSTANCE.createSubprocess();
			subprocess.setName(df.getVisibleName());
			subprocessMap.put(df.getIdName(), subprocess);
		}
		return subprocessMap.get(df.getIdName());
	}
	

	private static InfoElement loadInfoElement(URI infoElementUri) throws UnknownRepositoryException {
		String uuid = infoElementUri.host();
		CDOSession session = IRepositoryManager.INSTANCE.openSession(uuid);
		CDOView view = session.openView();
		InfoElement infoElement = (InfoElement) view.getResource(infoElementUri.path()).getEObject(infoElementUri.fragment());
		InfoElement infoElementCopy = EcoreUtil.copy(infoElement);
		view.close();
		session.close();
		return infoElementCopy;
	}
	
	/**
	 * Generates all the DITA resources selected by the
	 * {@link DocumentFeatureModelConfiguration} passed as an argument. All files
	 * will be generated inside the directory {@link File} designated by
	 * <code>destinationDir</code>
	 * 
	 * @param process
	 * @param destinationDir
	 * @throws IOException
	 */
	public static void generateDita(DocumentFeatureModelConfiguration dfmc, File destinationDir) throws IOException {
		TransformerUtil.generateDita(TransformerUtil.transformDfmconfToProcess(dfmc), destinationDir);
	}
	
	/**
	 * Generates all the DITA resources contained in the {@link Process} passed as
	 * an argument. All files will be generated inside the directory {@link File}
	 * designated by <code>destinationDir</code>
	 * 
	 * @param process
	 * @param destinationDir
	 * @throws IOException
	 */
	public static void generateDita(Process process, File destinationDir) throws IOException {
		
		Path destinationDirPath = destinationDir.toPath();
		
		HashMap<VariableIEContents,AttributeConfiguration> mapVarAttr = new HashMap<VariableIEContents,AttributeConfiguration>();
		
		try {
			DocumentBuilder docBuilder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
			
	        Document doc = docBuilder.newDocument();
	        DocumentType documentType = doc.getImplementation().createDocumentType("map", "-//OASIS//DTD DITA Map//EN", "map.dtd");
	        doc.appendChild(documentType);	        
	        Element map = doc.createElement("map");
	        doc.appendChild(map);
	        Element title = doc.createElement("title");
	        map.appendChild(title);
	        title.setTextContent(process.getName());
			Element mapRef = doc.createElement("mapref");
			mapRef.setAttribute("href", VARIABLES_DITAMAP_FILENAME);
			mapRef.setAttribute("format", "ditamap");
	        map.appendChild(mapRef);
	        
	        Document varDoc = docBuilder.newDocument();
	        DocumentType varDocumentType = varDoc.getImplementation().createDocumentType("map", "-//OASIS//DTD DITA Map//EN", "map.dtd");
		    varDoc.appendChild(varDocumentType);	        
	        Element varMap = varDoc.createElement("map");
	        varDoc.appendChild(varMap);
	        Element varTitle = varDoc.createElement("title");
	        varMap.appendChild(varTitle);
	        varTitle.setTextContent("DITA Map variables: " + process.getName());
	        
			for (Activity activity: filter(process.getNodes(), Activity.class)) {
					fillmapVarHashmap(activity, mapVarAttr);
			}
			
			for (FlowNode node: process.getNodes()) {
				if (node instanceof Activity) {
					navigateActivityChilds(destinationDir, (Activity) node, map, mapVarAttr);
				}
			}
			
			createVariableDitaMap(getMapVarAttrToMapString(mapVarAttr), varMap);
		
			FileOutputStream ditaOutputStream = new FileOutputStream(destinationDirPath.resolve(MAIN_DITAMAP_FILENAME).toFile());
			Serializer serializer = new Serializer(ditaOutputStream);
			serializer.setIndent(4);
			serializer.write(DOMConverter.convert(doc));
			ditaOutputStream.close();
			
			FileOutputStream varsOutputStream = new FileOutputStream(destinationDirPath.resolve(VARIABLES_DITAMAP_FILENAME).toFile());
			Serializer varSerializer = new Serializer(varsOutputStream);
			varSerializer.setIndent(4);
			varSerializer.write(DOMConverter.convert(varDoc));
			varsOutputStream.close();
		} catch (ParserConfigurationException e) {
			throw new IOException("Cannot create XML file", e);
		}
	}
	
	private static void fillmapVarHashmap(Activity activity, HashMap<VariableIEContents, AttributeConfiguration> mapVarAttr){
		
		InfoElement ie = activity.getInfoElement();
		DocumentFeatureSelection dfs = activity.getDocumentFeatureSelection(); 
		
		if(ie != null){
			for(VariableIEContents varIe : ie.getVariables()){
				boolean isFound = false;
				for (VariableAttributeConfiguration attr : findDocumentFeatureModelConfiguration(dfs).getGlobalVariableAttributes()) {
					for(Reference ref : attr.getReferences()){
						if(varIe.getIdName().equals(ref.getReferenceName()) 
						  && ie.getUuid().toString().equals(ref.getInfoElementURI().toString().split("#")[3])){
							isFound = true;
							mapVarAttr.put(varIe, attr);
						}
						if(isFound) break;
					}
					if(isFound) break;
				}
				if(!isFound){
					for(VariableAttributeConfiguration attr : dfs.getVariableAttributesConfiguration()) {
						for(Reference ref : attr.getReferences()){
							if(varIe.getIdName().equals(ref.getReferenceName()) 
							  && ie.getUuid().toString().equals(ref.getInfoElementURI().toString().split("#")[3])){
								isFound = true;
								mapVarAttr.put(varIe, attr);
							}
							if(isFound) break;
						}
						if(isFound) break;
					}				
					navigateParents(activity, varIe, ie, mapVarAttr);
				}
			}	
		}
		
		
		if (activity instanceof Process) {
			Process process = (Process) activity;
			for (Activity a : filter(process.getNodes(), Activity.class)) {
				fillmapVarHashmap(a, mapVarAttr);
			}
		}
		
	}
	
	private static DocumentFeatureModelConfiguration findDocumentFeatureModelConfiguration(DocumentFeatureSelection dfs) {
		EObject parent = dfs.eContainer();
		while (parent != null && !DocumentFeatureModelConfiguration.class.isInstance(parent)) {
			parent = parent.eContainer();
		}
		return (DocumentFeatureModelConfiguration) parent;
	}
	
	private static HashMap<String,String> getMapVarAttrToMapString(HashMap<VariableIEContents, AttributeConfiguration> mapVarAttr) {
		
		HashMap<String,String> mapString = new HashMap<String,String>();
		
		Iterator<VariableIEContents> iterator = mapVarAttr.keySet().iterator();
		while(iterator.hasNext()){
			VariableIEContents var = iterator.next();
			VariableAttributeConfiguration attr = (VariableAttributeConfiguration) mapVarAttr.get(var);
			if(mapString.get(attr.getAttribute().getIdName()) == null){
				mapString.put(attr.getAttribute().getIdName(), attr.getValue());	
			}
		}	
		return mapString;
	}

	private static void navigateParents(Activity activity, VariableIEContents varIe, InfoElement ie, HashMap<VariableIEContents, AttributeConfiguration> mapVarAttr) {
		
		FlowNode node = (FlowNode) activity;
		DocumentFeatureSelection dfs = node.getOwner().getDocumentFeatureSelection();
		
		if(dfs != null){
			boolean isFound = false;
			for(VariableAttributeConfiguration attr : dfs.getVariableAttributesConfiguration()){
				for(Reference ref : attr.getReferences()){
					if(varIe.getIdName().equals(ref.getReferenceName()) 
					  && ie.getUuid().toString().equals(ref.getInfoElementURI().toString().split("#")[3])){
						isFound = true;
						mapVarAttr.put(varIe, attr);
					}
					if(isFound) break;
				}
				if(isFound) break;
			}
			
			Process process = node.getOwner();
			Activity activityParent = (Activity) process;
			navigateParents(activityParent, varIe, ie, mapVarAttr);
		}	
	}
	

	private static void createVariableDitaMap(HashMap<String,String> mapString, Element element){
		mapString.keySet().forEach(key -> {
			String value =  mapString.get(key);
			createVariableNode(key, value, element);
		});
	}
	
	private static void createVariableNode(String key, String value, Element element){
		Element keyDef = element.getOwnerDocument().createElement("keydef");
		keyDef.setAttribute("keys", key);
		Element topicMeta = element.getOwnerDocument().createElement("topicmeta");
		Element keywords = element.getOwnerDocument().createElement("keywords");
		Element keyword = element.getOwnerDocument().createElement("keyword");
		keyword.setTextContent(value);
		keywords.appendChild(keyword);
		topicMeta.appendChild(keywords);
		keyDef.appendChild(topicMeta);
		element.appendChild(keyDef);
		
	}
	
	private static void navigateActivityChilds(File workingDir, Activity activity, Element element, HashMap<VariableIEContents, AttributeConfiguration> mapVarAttr) throws IOException {
		
		Path workingDirPath = workingDir.getAbsoluteFile().toPath();
		Files.createDirectories(workingDirPath);
		
		Element childElement = element.getOwnerDocument().createElement("topicref");
		element.appendChild(childElement);
		childElement.setAttribute("navtitle", activity.getName());
		
		if (activity.getInfoElement() != null) {
			InfoElement infoElement = activity.getInfoElement();
			Path ditaFilePath = workingDirPath.resolve(infoElement.getUuid().toString() + ".dita");
			
			if (infoElement instanceof DitaRepresentable) {
				childElement.setAttribute("href", workingDirPath.relativize(ditaFilePath).toString());

				if(infoElement instanceof TextIE) {
					String content = ((TextIE) infoElement).getContents();
					for(VariableIEContents varIe : infoElement.getVariables()){
						if (mapVarAttr.get(varIe) != null) {
							String attrRef = ((AttributeConfiguration) mapVarAttr.get(varIe)).getAttribute().getIdName();
							content = content.replace("[var=\"" + varIe.getIdName() + "\"]","<keyword keyref=\"" + attrRef + "\"/>" );
							((TextIE) infoElement).setContents(content);
						}
					}
				}
				
				DitaRepresentable ditaRepresentable = (DitaRepresentable) infoElement;
				try {
					FileOutputStream fileOutputStream = new FileOutputStream(ditaFilePath.toFile());
					Serializer serializer = new Serializer(fileOutputStream);
					serializer.setIndent(4);
					serializer.setMaxLength(80);
					serializer.write(DOMConverter.convert(ditaRepresentable.asDita(workingDirPath.toString())));
					fileOutputStream.close();
				} catch (IOException e) {
					DplfwPlugin.log(e);
				} catch (ParserConfigurationException e) {
					DplfwPlugin.log(e);
				}
			}
		}
		if (activity instanceof Process) {
			Process process = (Process) activity;
			for (Activity a :  filter(process.getNodes(), Activity.class)) {
				navigateActivityChilds(workingDir, a, childElement, mapVarAttr);
			}
		}
	}
	
	@SuppressWarnings("unchecked")
	private static <T> List<T> filter(List<?> input, Class<T> type) {
		List<T> result = new ArrayList<T>();
		for (Object eObject : input) {
			if (type.isInstance(eObject)) {
				result.add((T) eObject);
			}
		}
		return result;
	}	

	@SuppressWarnings("unchecked")
	private static <T> List<T> collectChildren(EObject eObj, Class<T> type) {
		List<T> result = new ArrayList<T>();
		if (type.isInstance(eObj)) {
			result.add((T) eObj);
		}
		for (Iterator<EObject> it = EcoreUtil.getAllContents(eObj, true); it.hasNext();) {
			EObject eObject = (EObject) it.next();
			if (type.isInstance(eObject)) {
				result.add((T) eObject);
			}
		}
		return result;
	}	
}
