package es.upv.dsic.issi.dplfw.core.util;

import es.upv.dsic.issi.dplfw.infoelements.InfoElement;

/**
 * Interface for clases implementing backend-dependent queries to look for
 * {@link InfoElement}s. Classes implementing this interface must be immutable.
 * As such, return values of the methods implemented by this interface must be
 * explicitly saved into a variable in order to use the produced query.
 * 
 * @author agomez
 *
 */
public interface IIEQueryBuilder {
	
	public static final class FACTORY {
		/**
		 * Returns a new query builder for the backend specified. It is noteworthy that
		 * classes implementing {@link IIEQueryBuilder} are immutable
		 * 
		 * @param backend
		 * @return
		 */
		public static IIEQueryBuilder createQueryFor(String backend) {
			switch (backend) {
				case "hibernate":
					return new HibernateIEQueryBuilder();
				case "hql":
					return new HibernateIEQueryBuilder();
				case "ocl":
					return new OCLIEQueryBuilder();
				case "sql":
					// The SQL query language is not supported because of the following reasons:
					// 1. The database only contains tables for the concrete classes, thus, it is not possible to
					//    query directly the InfoElements, but it has to be done separately for each subtype (TextIE,
					//    LocationIE, ImageIE, LinkIE, etc. This would require to modify the queries each time a new 
					//    subtype is added
					// 2. Tables are only created when an element of its type is created, thus, for examples if no
					//    LocationIE elements have been created the table INFOELEMENT_LOCATIONIE does not exist. 
					//    Thus, a query on that table would fail.
					// 3. Queries are extremely dependent on the database mapping configured, even table names may
					//    change, this is extremely undesirable and any solution would be very fragile.
					//
					// For these reasons, among others, no native queries for the generic DB mapping are implemented,
					// and que use the OCL query language instead
					return new OCLIEQueryBuilder();
				default: 
					return new OCLIEQueryBuilder();
			}
		}
	};
	
	String language();

	IIEQueryBuilder where();
	
	IIEQueryBuilder and();

	IIEQueryBuilder or();
	
	IIEQueryBuilder containsSubstring(String property, String str);

	IIEQueryBuilder contains(String property, String str);

	IIEQueryBuilder is(String property, String str);

	String build();	
}
