package es.upv.dsic.issi.dplfw.core.model.internal;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceVisitor;
import org.eclipse.core.resources.ProjectScope;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.SubMonitor;
import org.eclipse.core.runtime.preferences.IEclipsePreferences;
import org.eclipse.core.runtime.preferences.IScopeContext;
import org.osgi.service.prefs.BackingStoreException;

import es.upv.dsic.issi.dplfw.core.DplfwPlugin;
import es.upv.dsic.issi.dplfw.core.model.IDfmConfFile;
import es.upv.dsic.issi.dplfw.core.model.IDfmFile;
import es.upv.dsic.issi.dplfw.core.model.IDplFile;
import es.upv.dsic.issi.dplfw.core.model.IDplFolder;
import es.upv.dsic.issi.dplfw.core.model.IDplProject;

public class DplProject implements IDplProject {

	
	private static String DFM_FOLDER 				= "dfm.folder"; // TODO: Consider delete this config  in the future in favor of using a derived value from the DFM_FILE
	private static String DFMCONF_FOLDERS 			= "dfmconf.folders";
	private static String WFM_FOLDERS 				= "wfm.folders";
	private static String RESOURCES_FOLDER 			= "resources.folder";
	private static String OUT_FOLDER 				= "out.folder";

	private static String DFM_FILE 					= "dfm.file";

	private static String DEFAULT_DFM_FOLDER 		= "model";
	private static String DEFAULT_DFMCONF_FOLDER 	= "configurations";
	private static String DEFAULT_WFM_FOLDER 		= "workflows";
	private static String DEFAULT_RESOURCES_FOLDER 	= "resources";
	private static String DEFAULT_OUT_FOLDER 		= "out";
	
	private static String DEFAUL_DFM_FILE 			= DEFAULT_DFM_FOLDER + "model." + IDplProject.DFM_MODEL_EXTENSION;

	private IProject project;

	public DplProject(IProject project) {
		this.project = project;
		ensureFoldersExist();
	}
	
	public void initializePreferences() {
		IEclipsePreferences preferences = getEclipsePreferences();
		preferences.put(DFM_FOLDER, DEFAULT_DFM_FOLDER);
		preferences.put(DFMCONF_FOLDERS, DEFAULT_DFMCONF_FOLDER);
		preferences.put(WFM_FOLDERS, DEFAULT_WFM_FOLDER);
		preferences.put(RESOURCES_FOLDER, DEFAULT_RESOURCES_FOLDER);
		preferences.put(OUT_FOLDER, DEFAULT_OUT_FOLDER);
		try {
			preferences.flush();
		} catch (BackingStoreException e) {
			DplfwPlugin.log(e);
		}
	}

	@Override
	public IDfmFile initializeDfm(String dfmFileName, IProgressMonitor monitor) throws CoreException, IOException {
		IDfmFile dfmFile = null;
		if (monitor == null) {
			monitor = new NullProgressMonitor();
		}
		try {
			monitor.beginTask("Initializing Document Feature Model", 3);
			
			IEclipsePreferences preferences = getEclipsePreferences();
			String dfmFolderName = preferences.get(DFM_FOLDER, DEFAULT_DFM_FOLDER);
			IFolder dfmFolder = project.getFolder(dfmFolderName);
			
			if (!dfmFolder.exists()) {
				dfmFolder.create(true, false, SubMonitor.convert(monitor, 1));
			} else {
				monitor.worked(1);
			}
			
			dfmFile = new DfmFile(dfmFolder.getFile(dfmFileName));
			dfmFile.initialize(SubMonitor.convert(monitor, 1));
			
			setDfmFile(dfmFile);
			
		} finally {
			monitor.done();
		}
		return dfmFile;
	}
	
	protected synchronized void ensureFoldersExist() {
		List<IDplFolder> dplFolders = new ArrayList<IDplFolder>();
		
		dplFolders.add(getDfmFolder());
		dplFolders.addAll(getDfmconfFolders());
		dplFolders.addAll(getWfmFolders());
		dplFolders.add(getResourcesFolder());
		dplFolders.add(getOutputFolder());
		
		for (IDplFolder dplFolder : dplFolders) {
			if (!dplFolder.getFolder().exists()) {
				try {
					dplFolder.getFolder().create(true, false, new NullProgressMonitor());
				} catch (CoreException e) {
					DplfwPlugin.log(e);
				}
			}
		}
	}

	public IEclipsePreferences getEclipsePreferences() {
		if (!DplProject.hasDplNature(this.project)) {
			return null;
		}
		IScopeContext context = new ProjectScope(getProject());
		IEclipsePreferences eclipsePreferences = context.getNode(DplfwPlugin.PLUGIN_ID);
		return eclipsePreferences;
	}


	/**
	 * Returns true if the given project is accessible and it has
	 * a java nature, otherwise false.
	 * @param project IProject
	 * @return boolean
	 */
	protected static boolean hasDplNature(IProject project) {
		try {
			return project.hasNature(DplfwPlugin.NATURE_ID);
		} catch (CoreException e) {
			// Ignore exception silently
		}
		return false;
	}
	
	@Override
	public IProject getProject() {
		return project;
	}
	
	@Override
	public IDplFolder getDfmFolder() {
		String path = getEclipsePreferences().get(DFM_FOLDER, DEFAULT_DFM_FOLDER);
		return new DplFolder(project.getFolder(path));

	}
	
	@Override
	public void setDfmFolder(IDplFolder folder) {
		IEclipsePreferences eclipsePreferences = getEclipsePreferences();
		eclipsePreferences.put(DFM_FOLDER, folder.toString());
		try {
			eclipsePreferences.flush();
		} catch (BackingStoreException e) {
			DplfwPlugin.log(e);
		}
	}
	
	@Override
	public boolean isDfmFolder(IDplFolder folder) {
		return folder.equals(getDfmFolder());
	}

	@Override
	public IDfmFile getDfmFile() {
		String path = getEclipsePreferences().get(DFM_FILE, DEFAUL_DFM_FILE);
		return new DfmFile(project.getFile(path));
	}
	
	@Override
	public void setDfmFile(IDfmFile file) {
		IEclipsePreferences eclipsePreferences = getEclipsePreferences();
		eclipsePreferences.put(DFM_FILE, file.getFile().getProjectRelativePath().toString());
		try {
			eclipsePreferences.flush();
		} catch (BackingStoreException e) {
			DplfwPlugin.log(e);
		}
	}
	
	
	@Override
	public List<IDplFolder> getDfmconfFolders() {
		List<IDplFolder> folders = new ArrayList<IDplFolder>();
		IEclipsePreferences preferences = getEclipsePreferences();
		String value = preferences.get(DFMCONF_FOLDERS, DEFAULT_DFMCONF_FOLDER);
		String[] folderNames = StringUtils.split(value, ",");
		for (String folderName : folderNames) {
			folders.add(new DplFolder(project.getFolder(folderName)));
		}
		return folders;
	}
	
	@Override
	public void setDfmconfFolders(List<IDplFolder> folders) {
		IEclipsePreferences eclipsePreferences = getEclipsePreferences();
		String string = StringUtils.join(folders, ",");
		eclipsePreferences.put(DFMCONF_FOLDERS, string);
		try {
			eclipsePreferences.flush();
		} catch (BackingStoreException e) {
			DplfwPlugin.log(e);
		}
	}
	
	@Override
	public boolean isDfmconfFolder(IDplFolder folder) {
		List<IDplFolder> dfmconfFolders = getDfmconfFolders();
		for (IDplFolder dplFolder : dfmconfFolders) {
			if (folder.equals(dplFolder)) 
				return true;
		}
		return false;
	}
	
//	@Override
//	public IDplFolder findFolder(IFolder folder) {
//		DplFolder dplFolder = new DplFolder(folder);
//		if (dplFolder.equals(getDfmFolder())) {
//			return getDfmFolder();
//		} else if (dplFolder.equals(getResourcesFolder())) {
//			return getResourcesFolder();
//		} else if (dplFolder.equals(getOutputFolder())) {
//			return getOutputFolder();
//		} else {
//			for (IDplFolder configFolder : getDfmconfFolders()) {
//				if (dplFolder.equals(configFolder)) {
//					return configFolder;
//				}
//			}
//			for (IDplFolder wfmFolder : getWfmFolders()) {
//				if (dplFolder.equals(wfmFolder)) {
//					return wfmFolder;
//				}
//			}
//		}
//		return null;
//	}
	
	@Override
	public List<IDfmConfFile> findDfmConfFiles() throws CoreException {
		final List<IDfmConfFile> configurationFiles = new ArrayList<IDfmConfFile>();
		IResourceVisitor visitor = new IResourceVisitor() {
			@Override
			public boolean visit(IResource resource) throws CoreException {
				if (IDplFile.isDfmConfFile(resource)) {
					configurationFiles.add(new DfmConfFile((IFile) resource));
				}
				return true;
			}
		};
		project.accept(visitor);
		return configurationFiles;
	}
	
	@Override
	public List<IDfmConfFile> findDfmConfFilesFor(final IDfmFile dfmFile) throws CoreException {
		final List<IDfmConfFile> configurationFiles = new ArrayList<IDfmConfFile>();
		IResourceVisitor visitor = new IResourceVisitor() {
			@Override
			public boolean visit(IResource resource) throws CoreException {
				if (IDplFile.isDfmConfFile(resource)) {
					IDfmConfFile dfmConfFile = new DfmConfFile((IFile) resource);
					if (dfmConfFile.getDfmFile().equals(dfmFile)) {
						configurationFiles.add(dfmConfFile);
					}
				}
				return true;
			}
		};
		project.accept(visitor);
		return configurationFiles;

	}

	@Override
	public List<IDplFolder> getWfmFolders() {
		List<IDplFolder> folders = new ArrayList<IDplFolder>();
		IEclipsePreferences preferences = getEclipsePreferences();
		String value = preferences.get(WFM_FOLDERS, DEFAULT_WFM_FOLDER);
		String[] folderNames = StringUtils.split(value, ",");
		for (String folderName : folderNames) {
			folders.add(new DplFolder(project.getFolder(folderName)));
		}
		return folders;
	}
	
	@Override
	public void setWfmFolders(List<IDplFolder> folders) {
		IEclipsePreferences eclipsePreferences = getEclipsePreferences();
		String string = StringUtils.join(folders, ",");
		eclipsePreferences.put(WFM_FOLDERS, string);
		try {
			eclipsePreferences.flush();
		} catch (BackingStoreException e) {
			DplfwPlugin.log(e);
		}
	}

	@Override
	public boolean isWfmFolder(IDplFolder folder) {
		List<IDplFolder> wfmFolders = getWfmFolders();
		for (IDplFolder wfmFolder : wfmFolders) {
			if (folder.equals(wfmFolder)) 
				return true;
		}
		return false;
	}

	@Override
	public IDplFolder getResourcesFolder() {
		String path = getEclipsePreferences().get(RESOURCES_FOLDER, DEFAULT_RESOURCES_FOLDER);
		return new DplFolder(project.getFolder(path));
	}
	
	@Override
	public void setResourcesFolder(IDplFolder folder) {
		IEclipsePreferences eclipsePreferences = getEclipsePreferences();
		eclipsePreferences.put(RESOURCES_FOLDER, folder.toString());
		try {
			eclipsePreferences.flush();
		} catch (BackingStoreException e) {
			DplfwPlugin.log(e);
		}
	}

	@Override
	public boolean isResourcesFolder(IDplFolder folder) {
		return folder.equals(getResourcesFolder());
	}

	@Override
	public IDplFolder getOutputFolder() {
		String path = getEclipsePreferences().get(OUT_FOLDER, DEFAULT_OUT_FOLDER);
		return new DplFolder(project.getFolder(path));
	}
	
	@Override
	public void setOutputFolder(IDplFolder folder) {
		IEclipsePreferences eclipsePreferences = getEclipsePreferences();
		eclipsePreferences.put(OUT_FOLDER, folder.toString());
		try {
			eclipsePreferences.flush();
		} catch (BackingStoreException e) {
			DplfwPlugin.log(e);
		}
	}

	@Override
	public boolean isOuputFolder(IDplFolder folder) {
		return folder.equals(getOutputFolder());
	}

	@Override
	public String toString() {
		return project.getName();
	}
	
	@Override
	public boolean equals(Object obj) {
		if (project == null) {
			return false;
		} else if (!(obj instanceof DplProject)) {
			return false;
		} else {
			IProject objProject = ((DplProject) obj).getProject();
			return project.equals(objProject);
		}
	}
	
	@Override
	public int hashCode() {
		int hash = 33;
		return hash + project.hashCode();
	}
	
	@Override
	public <T> T getAdapter(Class<T> adapter) {
		return Platform.getAdapterManager().getAdapter(project, adapter);
	}
}
