package es.upv.dsic.issi.dplfw.core.ltk.move;

import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Path;
import org.eclipse.ltk.core.refactoring.Change;
import org.eclipse.ltk.core.refactoring.CompositeChange;
import org.eclipse.ltk.core.refactoring.RefactoringStatus;
import org.eclipse.ltk.core.refactoring.participants.CheckConditionsContext;
import org.eclipse.ltk.core.refactoring.participants.MoveParticipant;
import org.eclipse.osgi.util.NLS;

import es.upv.dsic.issi.dplfw.core.DplfwPlugin;
import es.upv.dsic.issi.dplfw.core.ltk.changes.ChangeDfmInDfmConfChange;
import es.upv.dsic.issi.dplfw.core.ltk.changes.ChangeDfmInDplProjectChange;
import es.upv.dsic.issi.dplfw.core.model.IDfmConfFile;
import es.upv.dsic.issi.dplfw.core.model.IDfmFile;
import es.upv.dsic.issi.dplfw.core.model.IDplFile;
import es.upv.dsic.issi.dplfw.core.model.IDplProject;
import es.upv.dsic.issi.dplfw.core.model.internal.DfmFile;
import es.upv.dsic.issi.dplfw.core.model.internal.DplProject;


/**
 * Performs actions to be carried out when a DFM is moved in a DPL project.
 * These actions are:
 * 1. Update the DPL project config, changing the DFM file location
 * 2. Update the DFM URI for all the DFMCONF files of the project whose URI matches 
 *    the moved DFM
 *   
 * @author agomez
 *
 */
public class MoveDfmParticipant extends MoveParticipant {

	private IDfmFile originDfmFile;
	private IDfmFile targetDfmFile;
	private List<IDfmConfFile> confFiles;
	private boolean isRegisteredDfm;
	
	public MoveDfmParticipant() {
	}

	@Override
	protected boolean initialize(Object element) {
		isRegisteredDfm = false;
		if (element instanceof IFile) {
			if (IDplFile.isDfmFile((IFile) element)) {
				originDfmFile = new DfmFile((IFile) element);
				if (getArguments().getDestination() instanceof IContainer) {
					IContainer targetContainer = (IContainer) getArguments().getDestination();
					IFile targetFile = targetContainer.getFile(new Path(originDfmFile.getFile().getName()));
					// It is safe to cast to IDfmFile since the filename is the same than the originDfmFile
					targetDfmFile = (IDfmFile) IDplFile.create(targetFile);
				}
				IProject project = originDfmFile.getDplProject().getProject();
				try {
					if (project.hasNature(DplfwPlugin.NATURE_ID)) {
						IDplProject dplProject = new DplProject(project);
						if (dplProject.getDfmFile().equals(originDfmFile)) {
							isRegisteredDfm = true;
						}
						confFiles = dplProject.findDfmConfFilesFor(originDfmFile);
						return true;
					}
				} catch (CoreException e) {
					DplfwPlugin.log(e);
				}
			}
		}
		return false;
	}

	@Override
	public String getName() {
		return "Move Document Feature Model";
	}

	@Override
	public RefactoringStatus checkConditions(IProgressMonitor pm, CheckConditionsContext context) throws OperationCanceledException {
		RefactoringStatus refactoringStatus = new RefactoringStatus();
		if (isRegisteredDfm) {
			refactoringStatus.addInfo(NLS.bind("File ''{0}'' is the default document feature model for project ''{1}''. Project configuration will be updated.",
					originDfmFile.toString(),
					originDfmFile.getDplProject().toString()));
		}
		for (IDfmConfFile confFile : confFiles) {
			refactoringStatus.addInfo(NLS.bind("Configuration file ''{0}'' depends on ''{1}'' and will be updated.", 
					confFile.toString(),
					originDfmFile.toString()));
		}
		return refactoringStatus;
	}

	
	@Override
	public Change createChange(IProgressMonitor pm) throws CoreException, OperationCanceledException {
		CompositeChange compositeChange = new CompositeChange(getName());
		for (IDfmConfFile confFile : confFiles) {
			Change change = new ChangeDfmInDfmConfChange(confFile, targetDfmFile);
			compositeChange.add(change);
		}
		Change change = new ChangeDfmInDplProjectChange(originDfmFile.getDplProject(), targetDfmFile);
		compositeChange.add(change);
		return compositeChange;
	}
}
