package es.upv.dsic.issi.dplfw.core.builder;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.resources.IResourceVisitor;
import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.SubMonitor;
import org.eclipse.osgi.util.NLS;

import es.upv.dsic.issi.dplfw.core.builder.runnables.RebuildDfmconfBuildAction;
import es.upv.dsic.issi.dplfw.core.builder.runnables.ValidateDplProjectBuildAction;
import es.upv.dsic.issi.dplfw.core.model.IDfmConfFile;
import es.upv.dsic.issi.dplfw.core.model.IDfmFile;
import es.upv.dsic.issi.dplfw.core.model.IDplFile;
import es.upv.dsic.issi.dplfw.core.model.IDplProject;


public class DplBuilder extends IncrementalProjectBuilder {



	class DplProjectBuilderVisitor implements IResourceDeltaVisitor, IResourceVisitor {

		/**
		 * Visitor for delta (partial build)
		 */
		@Override
		public boolean visit(IResourceDelta delta) throws CoreException {
			IResource resource = delta.getResource();
			switch (delta.getKind()) {
				case IResourceDelta.ADDED:
					visitResource(resource);
					break;
				case IResourceDelta.REMOVED:
					break;
				case IResourceDelta.CHANGED:
					visitResource(resource);
					break;
				default:
			}
			return true; //return true to continue visiting children
		}
		
		/**
		 * Visitor for resource (full build)
		 */
		@Override
		public boolean visit(IResource resource) throws CoreException {
			visitResource(resource);
			return true; //return true to continue visiting children
		}
		
		private void visitResource(IResource resource) throws CoreException {
			if (resource instanceof IFile && IDplFile.isDfmFile(resource)) {
				IDplProject dplProject = IDplProject.create(resource.getProject());
				IDfmFile dfmFile = IDplFile.create((IFile)resource).as(IDfmFile.class);
				List<IDfmConfFile> dfmConfFiles = dplProject.findDfmConfFilesFor(dfmFile);
				for (IDfmConfFile dfmConfFile : dfmConfFiles) {
					buildActions.add(new RebuildDfmconfBuildAction(dfmConfFile));
				}
			} else if (resource instanceof IProject) {
				buildActions.add(new ValidateDplProjectBuildAction((IProject) resource));
			}
		}
	};
	
	private List<IBuildActionWithProgressMonitor> buildActions;
	
	@SuppressWarnings("rawtypes")
	protected IProject[] build(int kind, Map args, IProgressMonitor monitor) throws CoreException {
		if (kind == FULL_BUILD) {
			fullBuild(monitor);
		} else {
			IResourceDelta delta = getDelta(getProject());
			if (delta == null) {
				fullBuild(monitor);
			} else {
				incrementalBuild(delta, monitor);
			}
		}
		return null;
	}

	/**
	 * Performs a full build.
	 * 
	 * First computes the required build actions by visiting all the resources of 
	 * the project, and second executes all the required build actions executing the 
	 * {@link #runBuildActions(IProgressMonitor)} method 
	 * 
	 * @param monitor
	 * @throws CoreException
	 */
	protected void fullBuild(IProgressMonitor monitor) throws CoreException {
		buildActions = new ArrayList<IBuildActionWithProgressMonitor>();
		getProject().accept(new DplProjectBuilderVisitor());
		runBuildActions(monitor);
	}

	/**
	 * Performs an incremental build.
	 * 
	 * First computes the required build actions by visiting all the resources of 
	 * the delta, and second executes all the required build actions executing the 
	 * {@link #runBuildActions(IProgressMonitor)} method.
	 * 
	 * The project validation is always executed.
	 * 
	 * @param delta
	 * @param monitor
	 * @throws CoreException
	 */
	protected void incrementalBuild(IResourceDelta delta, IProgressMonitor monitor) throws CoreException {
		buildActions = new ArrayList<IBuildActionWithProgressMonitor>();
		delta.accept(new DplProjectBuilderVisitor());
		runBuildActions(monitor);
	}


	/**
	 * Runs all the required build actions stored in {@link #buildActions}
	 * 
	 * @param monitor
	 * @throws CoreException
	 */
	protected void runBuildActions(IProgressMonitor monitor) throws CoreException {
		if (monitor == null) { monitor = new NullProgressMonitor(); }
		try {
			int workUnits = 0;
			for (IBuildActionWithProgressMonitor buildAction : buildActions) {
				workUnits += buildAction.workUnits();
			}
			monitor.beginTask("", workUnits); 
			for (IBuildActionWithProgressMonitor buildAction : buildActions) {
				monitor.setTaskName(buildAction.getDescription());
				buildAction.run(SubMonitor.convert(monitor, buildAction.workUnits()));
				if (monitor.isCanceled()) {
					throw new OperationCanceledException();
				}
			}
		} finally {
			monitor.done();
		}
	}
	
	/**
	 * Cleans all the problem markers from the project and its contents
	 * in which this builder is registered
	 */
	@Override
	protected void clean(IProgressMonitor monitor) throws CoreException {
		if (monitor == null) { monitor = new NullProgressMonitor(); }
		try {
			IProject project = getProject();
			monitor.beginTask(NLS.bind("Cleaning \"%s\" project", project.getName()), IProgressMonitor.UNKNOWN);
			project.deleteMarkers(IMarker.PROBLEM, true, IResource.DEPTH_INFINITE);
		} finally {
			monitor.done();
		}
	}
}
