package es.upv.dsic.issi.dplfw.core.ui.util;

import java.awt.GraphicsEnvironment;
import java.io.InputStream;
import java.lang.reflect.Method;
import java.util.Locale;

import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Display;


public class SWTUtil {
	
	/**
	 * Creates a new resized {@link Image} (up to <code>maxWidth</code>) using
	 * the given {@link Image} as input. <b>The caller is responsible of disposing
	 * the new {@link Image}</b>
	 * 
	 * @param stream
	 *            the {@link InputStream} of the source image
	 * @param maxWidth
	 *            the max width of the resulting {@link Image}
	 * @return a new {@link Image}
	 */
	
	public static Image createResizedImage(Image sourceImage, int maxWidth) {
		int width = sourceImage.getImageData().width;
		float scale = (float) maxWidth / width;
		int height = (int) (sourceImage.getImageData().height * scale);
		return createResizedImage(sourceImage, (maxWidth < width ? maxWidth : width), height);
	}
	
	/**
	 * Creates a new resized {@link Image} using the given {@link Image} as
	 * input. <b>The caller is responsible of disposing the new {@link Image}</b>
	 * 
	 * @param stream
	 *            the {@link InputStream} of the source image
	 * @param width
	 *            the max width of the resulting {@link Image}
	 * @param height
	 *            the max width of the resulting {@link Image}
	 * @return a new {@link Image}
	 */
	
	public static Image createResizedImage(Image sourceImage, int width, int height) {
		Image resizedImage = new Image(Display.getDefault(),
				width  > 0 ? width  : 1,
				height > 0 ? height : 1);
		
		GC gc = new GC(resizedImage);
		gc.setAntialias(SWT.ON);
		gc.drawImage(sourceImage, 0, 0, sourceImage.getImageData().width,
				sourceImage.getImageData().height, 0, 0, width, height);
		gc.dispose();
		return resizedImage;
	}
	
	
	/**
	 * Creates a new dummy {@link image} of the specified size using the given color.
	 * <b>The caller is responsible of disposing the new {@link image}</b>
	 * 
	 * @param stream
	 *            the {@link InputStream} of the source image
	 * @param maxWidth
	 *            the max width of the resulting {@link Image}
	 * @return a new {@link Image}
	 */
	
	public static Image createEmptyImage(Color color, int width, int height) {
		Image resizedImage = new Image(Display.getDefault(),
				width  > 0 ? width  : 1,
				height > 0 ? height : 1);
		GC gc = new GC(resizedImage);
		gc.setAntialias(SWT.ON);
		gc.setBackground(color);
		gc.fillRectangle(0, 0, width, height);
		gc.dispose();
		return resizedImage;
	}
	
	/**
	 * Returns the SWT font for the given logical font name or 
	 * <code>null</code> if no font is found for the given name
	 * @param logicalName
	 * @param size
	 * @param style
	 * @return
	 */
	protected static Font getLogicalFont(String logicalName, int size, int style) {
		String key = String.format("%s.%s.%d.%d", "es.upv.dsic.issi.dplfw.core.commonlibs", logicalName, size, style);
	    if (!JFaceResources.getFontRegistry().hasValueFor(key)) {
	    	String defaultSerifFontName = findSystemFontNameFromAWTLogicalName(logicalName);
	    	if (defaultSerifFontName != null) {
		    	Font font = new Font(Display.getDefault(), defaultSerifFontName, size, style);
		    	JFaceResources.getFontRegistry().put(key, font.getFontData());
		    	font.dispose();
	    	}
	    }
	    return JFaceResources.getFontRegistry().get(key);
	}
	
	public static Font getDefaultTextFont() {
		Font font = getLogicalFont(java.awt.Font.SERIF, 12, SWT.NONE);
		if (font != null) { 
			return font;
		} else {
			return JFaceResources.getTextFont();
		}
	}

	public static Font getDefaultTitleFont() {
		Font font = getLogicalFont(java.awt.Font.SERIF, 18, SWT.BOLD);
		if (font != null) { 
			return font;
		} else {
			return JFaceResources.getHeaderFont();
		}
	}
	
	/**
	 * Finds the system font name mapped to the given AWT logical name
	 * @param logicalName
	 * @return
	 */
	private static String findSystemFontNameFromAWTLogicalName(String logicalName) {
		try {
			// First we get all fonts to ensure that the FontManager is loaded
			GraphicsEnvironment.getLocalGraphicsEnvironment().getAllFonts();
			// And now the dirty hack :(
			// We use internal methods of the SUN implementation of the JRE 
			// to query the registered fonts using the reflective API
			Class<?> fontManagerClass = Class.forName("sun.font.FontManager");
			Method getRegisteredFontsMethod = fontManagerClass.getMethod("getRegisteredFonts", new Class<?>[] {});
			Object[] registeredFonts = (Object[]) getRegisteredFontsMethod.invoke(fontManagerClass, new Object[] {});
			Class<?> font2DClass = Class.forName("sun.font.Font2D");
			for (Object fontObject : registeredFonts) {
				Method getNameMethod = font2DClass.getMethod("getFamilyName", Locale.class);
				String familyName = (String) getNameMethod.invoke(fontObject, Locale.getDefault());
				Class<?> compositeFontClass = Class.forName("sun.font.CompositeFont");
				if (familyName.equals(logicalName) && compositeFontClass.isInstance(fontObject)) {
					Method getSlotFontMethod = compositeFontClass.getMethod("getSlotFont", Integer.TYPE);
					Object physicalFont = getSlotFontMethod.invoke(fontObject, 0);
					return (String) getNameMethod.invoke(physicalFont, Locale.getDefault());
				}
			}
		} catch (Exception e) {
		} 
		return null;
	}
	
}
