package es.upv.dsic.issi.dplfw.core.ui.project.properties;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map.Entry;

import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.jface.layout.TreeColumnLayout;
import org.eclipse.jface.preference.PreferencePage;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.AbstractTreeViewer;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.ColumnViewer;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.ComboBoxCellEditor;
import org.eclipse.jface.viewers.DecorationOverlayIcon;
import org.eclipse.jface.viewers.EditingSupport;
import org.eclipse.jface.viewers.IDecoration;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ILabelProviderListener;
import org.eclipse.jface.viewers.LabelProviderChangedEvent;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.TreeViewerColumn;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.ui.dialogs.PropertyPage;
import org.eclipse.ui.model.WorkbenchContentProvider;
import org.eclipse.ui.model.WorkbenchLabelProvider;

import es.upv.dsic.issi.dplfw.core.model.IDplFolder;
import es.upv.dsic.issi.dplfw.core.model.IDplProject;
import es.upv.dsic.issi.dplfw.core.ui.DplfwUiPlugin;
import es.upv.dsic.issi.dplfw.core.ui.filters.FoldersFilter;
import es.upv.dsic.issi.dplfw.core.ui.filters.HiddenFilesFilter;

public class DplfwRootPropertyPage extends PropertyPage {

	/**
	 * Wrapper class to provide the functionality of a WorkbenchLabelProvider in a ColumnLabelProvider
	 * @author agomez
	 *
	 */
	private final class DecoratingWorkbenchColumnLabelProvider extends ColumnLabelProvider {
		ILabelProvider labelProvider = new WorkbenchLabelProvider();
		/**
		 * The Decorations of the DecoratingWorkbenchaLabelProvider are computed with a delay,
		 * we need to force update the viewer labels when the decorations are applied
		 */
		ILabelProviderListener listener = new ILabelProviderListener() {
			@Override
			public void labelProviderChanged(LabelProviderChangedEvent event) {
				viewer.update(event.getElements(), null);
			}
		};
		
		public DecoratingWorkbenchColumnLabelProvider() {
			labelProvider.addListener(listener);
		}
		
		@Override
		public String getText(Object element) {
			return labelProvider.getText(element);
		}
		
		@Override
		public Image getImage(Object element) {
			Image image = labelProvider.getImage(element);
			// We apply again the DplfolderDecorator using the local values 
			// to represent the dialog's state instead of the Workspace state
			if (element instanceof IFolder) {
				IDplFolder dplFolder = IDplFolder.create((IFolder) element);
				String key = null;
				if (DFM_FOLDER_LABEL.equals(folders.get(dplFolder))) {
					key = DplfwUiPlugin.MODEL_FOLDER_OVERLAY;
				} else if (DFMCONF_FOLDER_LABEL.equals(folders.get(dplFolder))) {
					key = DplfwUiPlugin.CONFIGS_FOLDER_OVERLAY;
				} else if (WFM_FOLDER_LABEL.equals(folders.get(dplFolder))) {
					key = DplfwUiPlugin.WORKFLOWS_FOLDER_OVERLAY;
				} else if (RESOURCES_FOLDER_LABEL.equals(folders.get(dplFolder))) {
					key = DplfwUiPlugin.RESOURCES_FOLDER_OVERLAY;
				} else if (OUTPUT_FOLDER_LABEL.equals(folders.get(dplFolder))) {
					key = DplfwUiPlugin.OUTPUT_FOLDER_OVERLAY;
				}  
				if (key != null) {
					ImageDescriptor descriptor = DplfwUiPlugin.getDefault().getImageRegistry().getDescriptor(key);
					DecorationOverlayIcon decoratedImage = new DecorationOverlayIcon(image, descriptor, IDecoration.TOP_RIGHT);
					return decoratedImage.createImage();
				}
			}
			return image;
		}
		
		@Override
		public void dispose() {
			labelProvider.removeListener(listener);
			labelProvider.dispose();
			super.dispose();
		}
	}

	/**
	 * Label provider for types of DPL folders
	 * @author agomez
	 *
	 */
	private final class ResourceTypeColumnLabelProvider extends ColumnLabelProvider {

		@Override
		public String getText(Object element) {
			if (element instanceof IFolder) {
				IDplFolder dplFolder = IDplFolder.create((IFolder) element);
				if (folders.containsKey(dplFolder)) {
					return folders.get(dplFolder);
				}
			}
			return NONE;
		}
	}

	

	/**
	 * Editing support to change folder types
	 * @author agomez
	 *
	 */
	private final class FolderTypeEditingSupport extends EditingSupport {
		private List<String> options = new ArrayList<String>();
		private FolderTypeEditingSupport(ColumnViewer viewer) {
			super(viewer);
			options.add(NONE);
			options.add(DFM_FOLDER_LABEL);
			options.add(DFMCONF_FOLDER_LABEL);
			options.add(WFM_FOLDER_LABEL);
			options.add(RESOURCES_FOLDER_LABEL);
			options.add(OUTPUT_FOLDER_LABEL);
		}

		@Override
		protected void setValue(Object element, Object value) {
			if (element instanceof IFolder) {
				IDplFolder dplFolder = IDplFolder.create((IFolder) element);
				if (value.equals(NONE)) {
					folders.remove(dplFolder);
				} else {
					folders.put(dplFolder, options.get((Integer) value));
				}
				viewer.update(element, null);
				updateErrors();
			}
		}

		@Override
		protected Object getValue(Object element) {
			if (element instanceof IFolder) {
				IDplFolder dplFolder = IDplFolder.create((IFolder) element);
				if (folders.containsKey(dplFolder)) {
					return options.indexOf(folders.get(dplFolder));
				}
			}
			return options.indexOf(NONE);
		}

		@Override
		protected CellEditor getCellEditor(Object element) {
			ComboBoxCellEditor cellEditor = new ComboBoxCellEditor(viewer.getTree(),
					options.toArray(new String[] {}), SWT.NONE);
			CCombo combo = (CCombo) cellEditor.getControl();
			combo.setEditable(false);
			combo.setVisibleItemCount(options.size());
			return cellEditor;
		}

		@Override
		protected boolean canEdit(Object element) {
			return true;
		}
	}

	private TreeViewer viewer;

	private static final String DFM_FOLDER_LABEL = "Document Feature Model folder";
	private static final String DFMCONF_FOLDER_LABEL = "Configurations folder";
	private static final String WFM_FOLDER_LABEL = "Workflows folder";
	private static final String RESOURCES_FOLDER_LABEL = "Resources folder";
	private static final String OUTPUT_FOLDER_LABEL = "Output folder";
	private static final String NONE = "<None>";
	
	private HashMap<IDplFolder, String> folders = new HashMap<IDplFolder, String>();
	
	/**
	 * Constructor for SamplePropertyPage.
	 */
	public DplfwRootPropertyPage() {
		super();
		noDefaultAndApplyButton();
	}

	/**
	 * @see PreferencePage#createContents(Composite)
	 */
	protected Control createContents(Composite parent) {
		// We initilize the fodlers here since the getElement() method which
		// defines the active project returns null when the constructor is executed
		
		initializeFolders();
		
		// Create the UI
		
		Composite composite = new Composite(parent, SWT.NONE);
		composite.setLayoutData(new GridData(GridData.FILL, GridData.FILL, true, true));
		GridLayout compositeLayout = new GridLayout();
		compositeLayout.marginWidth = 0;
		compositeLayout.marginHeight = 0;
		composite.setLayout(compositeLayout);

		Label viewerLabel = new Label(composite, SWT.NONE);
		viewerLabel.setLayoutData(new GridData());
		viewerLabel.setText("Select the special folders for this Document Product Line project");
		
		Composite treeComposite = new Composite(composite, SWT.NONE);
		treeComposite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		
		Tree tree = new Tree(treeComposite, SWT.SINGLE | SWT.BORDER | SWT.FULL_SELECTION);
		tree.setHeaderVisible(true);
		tree.setLinesVisible(true);
		tree.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		
		TreeColumnLayout layout = new TreeColumnLayout();
		treeComposite.setLayout(layout);
		
		viewer = new TreeViewer(tree);

		TreeViewerColumn folderColumn = new TreeViewerColumn(viewer, SWT.NONE);
		layout.setColumnData(folderColumn.getColumn(), new ColumnWeightData(75, 150));
		folderColumn.getColumn().setText("Folder");
		folderColumn.setLabelProvider(new DecoratingWorkbenchColumnLabelProvider());
		
		TreeViewerColumn typeColumn = new TreeViewerColumn(viewer, SWT.NONE);
		layout.setColumnData(typeColumn.getColumn(), new ColumnWeightData(25, 200));
		typeColumn.getColumn().setText("Type");
		typeColumn.setLabelProvider(new ResourceTypeColumnLabelProvider());
		EditingSupport editingSupport = new FolderTypeEditingSupport(viewer);
		typeColumn.setEditingSupport(editingSupport);
		
		viewer.setContentProvider(new WorkbenchContentProvider());
		viewer.setAutoExpandLevel(AbstractTreeViewer.ALL_LEVELS);
		viewer.setInput(getProject());
		viewer.setFilters(new ViewerFilter[] { new FoldersFilter(), new HiddenFilesFilter() });

		
		return composite;

	}

	protected void initializeFolders() {
		IProject project = (IProject) getElement().getAdapter(IProject.class);
		IDplProject dplProject = IDplProject.create(project);
		folders.put(dplProject.getDfmFolder(), DFM_FOLDER_LABEL);
		for (IDplFolder folder : dplProject.getDfmconfFolders()) {
			folders.put(folder, DFMCONF_FOLDER_LABEL);
		}
		for (IDplFolder folder : dplProject.getWfmFolders()) {
			folders.put(folder, WFM_FOLDER_LABEL);
		}
		folders.put(dplProject.getResourcesFolder(), RESOURCES_FOLDER_LABEL);
		folders.put(dplProject.getOutputFolder(), OUTPUT_FOLDER_LABEL);
	}


	protected void updateErrors() {
		if (!folders.containsValue(DFM_FOLDER_LABEL)) {
			setErrorMessage("No Document Feature Model folder has been specified");
			setValid(false);
			return;
		} else if (!folders.containsValue(DFMCONF_FOLDER_LABEL)) {
			setErrorMessage("At least one folder is needed to store Document Feature Model Configurations");
			setValid(false);
			return;
		} else if (!folders.containsValue(WFM_FOLDER_LABEL)) {
			setErrorMessage("At least one folder is needed to store document workflows");
			setValid(false);
			return;
		} else if (!folders.containsValue(RESOURCES_FOLDER_LABEL)) {
			setErrorMessage("No resources folder has been specified");
			setValid(false);
			return;
		} else if (!folders.containsValue(OUTPUT_FOLDER_LABEL)) {
			setErrorMessage("No output folder has been specified");
			setValid(false);
			return;
		} else if (findFoldersByType(DFM_FOLDER_LABEL).size() > 1) {
			setErrorMessage("Only one folder can be selected as the Document Feature Model file folder");
			setValid(false);
			return;
		} else if (findFoldersByType(RESOURCES_FOLDER_LABEL).size() > 1) {
			setErrorMessage("Only one folder can be selected as the resources folder");
			setValid(false);
			return;
		} else if (findFoldersByType(OUTPUT_FOLDER_LABEL).size() > 1) {
			setErrorMessage("Only one folder can be selected as the output folder");
			setValid(false);
			return;
		}
		setErrorMessage(null);
		setValid(true);
		return;
	}

	protected List<IDplFolder> findFoldersByType(String type) {
		List<IDplFolder> returnFolders = new ArrayList<IDplFolder>();
		for (Entry<IDplFolder, String> entry : folders.entrySet()) {
			if (entry.getValue().equals(type)) {
				returnFolders.add(entry.getKey());
			}
		}
		return returnFolders;		
	}
	
	protected void performDefaults() {
		super.performDefaults();
	}
	
	public boolean performOk() {
		try {
			IDplProject  dplProject = IDplProject.create(getProject());
			dplProject.setDfmFolder(findFoldersByType(DFM_FOLDER_LABEL).get(0));
			dplProject.setDfmconfFolders(findFoldersByType(DFMCONF_FOLDER_LABEL));
			dplProject.setWfmFolders(findFoldersByType(WFM_FOLDER_LABEL));
			dplProject.setResourcesFolder(findFoldersByType(RESOURCES_FOLDER_LABEL).get(0));
			dplProject.setOutputFolder(findFoldersByType(OUTPUT_FOLDER_LABEL).get(0));
			return true;
		} catch (Exception e) {
			return false;
		}
	}	

	protected IProject getProject() {
		return (IProject) getElement().getAdapter(IProject.class);
	}
	
}